import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers in the cabinet, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation:
        # 'item1', 'item2', 'item3' (tomatoes), 'plate'
        # Drawer positions: 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # --- Step 1: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_side_pos' in object positions.")
            shutdown_environment(env)
            return

        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer grasp) ---
        # Get current gripper position and set orientation to 90 deg about z
        # We'll assume the drawer handle requires a rotation about z axis
        # Use scipy.spatial.transform.R to get quaternion for 90 deg about z
        from scipy.spatial.transform import Rotation as R
        quat_ninety_deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        obs, reward, done = rotate(env, task, target_quat=quat_ninety_deg)
        if done:
            print("[Task] Task ended after rotate to 90 deg!")
            return

        # --- Step 3: Move gripper to the anchor position of the bottom drawer ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_anchor_pos' in object positions.")
            shutdown_environment(env)
            return

        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pull the drawer open ---
        # Pull along x axis (assume drawer opens along +x, adjust if needed)
        # Use a reasonable pull distance (e.g., 0.15m)
        pull_distance = 0.15
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # --- Step 5: Move gripper above item1 (first tomato) ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] Could not find 'item1' in object positions.")
            shutdown_environment(env)
            return

        obs, reward, done = move(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # --- Step 6: Pick item1 ---
        obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking item1!")
            return

        # --- Step 7: Move gripper above plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] Could not find 'plate' in object positions.")
            shutdown_environment(env)
            return

        obs, reward, done = move(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after move to plate!")
            return

        # --- Step 8: Place item1 on plate ---
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing item1!")
            return

        # --- Step 9: Move gripper above item2 (second tomato) ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] Could not find 'item2' in object positions.")
            shutdown_environment(env)
            return

        obs, reward, done = move(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Task ended after move to item2!")
            return

        # --- Step 10: Pick item2 ---
        obs, reward, done = pick(env, task, target_pos=item2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking item2!")
            return

        # --- Step 11: Move gripper above plate (again) ---
        obs, reward, done = move(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after move to plate (2nd time)!")
            return

        # --- Step 12: Place item2 on plate ---
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing item2!")
            return

        # --- (Extra) Move and place item3 (third tomato) if present ---
        if 'item3' in positions:
            try:
                item3_pos = positions['item3']
                obs, reward, done = move(env, task, target_pos=item3_pos)
                if done:
                    print("[Task] Task ended after move to item3!")
                    return
                obs, reward, done = pick(env, task, target_pos=item3_pos, approach_distance=0.15, approach_axis='z')
                if done:
                    print("[Task] Task ended after picking item3!")
                    return
                obs, reward, done = move(env, task, target_pos=plate_pos)
                if done:
                    print("[Task] Task ended after move to plate (3rd time)!")
                    return
                obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
                if done:
                    print("[Task] Task ended after placing item3!")
                    return
            except Exception as e:
                print(f"[Warning] Could not complete item3 transfer: {e}")

        print("[Task] All tomatoes placed on plate. Task completed successfully! Reward:", reward)

    except Exception as e:
        print("[Exception] An error occurred during task execution:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()