import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Pull open any drawer that is not locked, then drop the 2 tomatoes onto the plate.
        # Specification: [pull], [pick], [place], [pick], [place]
        # We assume 'item1' and 'item2' are the tomatoes, 'plate' is the destination.
        # For the drawer: open any unlocked drawer. From the initial state, 'bottom', 'middle', and 'top' drawers exist.
        # 'top' and 'middle' are locked, so only 'bottom' is not locked.

        # --- Step 1: Pull open the bottom drawer ---
        # To pull, we must:
        #   - Move gripper to the anchor position of the bottom drawer
        #   - Rotate gripper to the correct orientation (usually 90 deg)
        #   - Move to side position if needed
        #   - Pick the drawer handle (simulate grasp)
        #   - Pull along the drawer axis

        # 1. Move to bottom_anchor_pos
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] 'bottom_anchor_pos' not found in object positions.")
            return

        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to bottom_anchor_pos: {e}")
            return

        # 2. Rotate gripper to 90 degrees (if not already)
        # We'll try to get the quaternion for 90 deg rotation about z axis
        try:
            from scipy.spatial.transform import Rotation as R
            quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            # RLBench/pyrep expects [x, y, z, w]
            obs, reward, done = rotate(
                env,
                task,
                target_quat=quat_90deg,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during rotate to 90 deg!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate to 90 deg: {e}")
            return

        # 3. Pick the drawer handle (simulate grasp)
        try:
            # Approach the anchor position and close gripper
            obs, reward, done = pick(
                env,
                task,
                target_pos=bottom_anchor_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick of drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick of drawer handle: {e}")
            return

        # 4. Pull the drawer open (along x axis, positive direction)
        try:
            # Estimate a reasonable pull distance (e.g., 0.15m)
            pull_distance = 0.15
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pull of drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull of drawer: {e}")
            return

        # --- Step 2: Pick item1 (tomato1) from table ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] 'item1' not found in object positions.")
            return

        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=item1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick of item1!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick of item1: {e}")
            return

        # --- Step 3: Place item1 on plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] 'plate' not found in object positions.")
            return

        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place of item1!")
                return
        except Exception as e:
            print(f"[Error] Exception during place of item1: {e}")
            return

        # --- Step 4: Pick item2 (tomato2) from table ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] 'item2' not found in object positions.")
            return

        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=item2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick of item2!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick of item2: {e}")
            return

        # --- Step 5: Place item2 on plate ---
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place of item2!")
                return
        except Exception as e:
            print(f"[Error] Exception during place of item2: {e}")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()