import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers in the cabinet, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object name mapping (from object list and environment) ---
        # Drawer positions
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
        bottom_side_pos = positions.get('bottom_side_pos', None)
        bottom_joint_pos = positions.get('bottom_joint_pos', None)
        middle_anchor_pos = positions.get('middle_anchor_pos', None)
        middle_side_pos = positions.get('middle_side_pos', None)
        middle_joint_pos = positions.get('middle_joint_pos', None)
        top_anchor_pos = positions.get('top_anchor_pos', None)
        top_side_pos = positions.get('top_side_pos', None)
        top_joint_pos = positions.get('top_joint_pos', None)
        # Waypoint for safe moves
        waypoint1 = positions.get('waypoint1', None)
        # Items (tomatoes)
        item1_pos = positions.get('item1', None)
        item2_pos = positions.get('item2', None)
        item3_pos = positions.get('item3', None)
        # Plate
        plate_pos = positions.get('plate', None)

        # --- Choose which drawer to open (bottom, middle, or top) ---
        # We'll open the bottom drawer for simplicity
        drawer_anchor_pos = bottom_anchor_pos
        drawer_side_pos = bottom_side_pos
        drawer_joint_pos = bottom_joint_pos
        drawer_name = 'bottom'

        # --- Step 1: Move gripper to the side position of the drawer (move) ---
        # This is a safe approach position before rotating
        if drawer_side_pos is not None:
            print(f"[Task] Moving to side position of {drawer_name} drawer.")
            obs, reward, done = move(env, task, target_pos=drawer_side_pos)
            if done:
                print("[Task] Task ended during move to drawer side position!")
                return
        else:
            print("[Error] Drawer side position not found!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (rotate) ---
        # Assume 90 degrees about z axis (vertical), quaternion for 90 deg about z: [0, 0, sqrt(0.5), sqrt(0.5)]
        target_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
        print(f"[Task] Rotating gripper to 90 degrees for drawer grasp.")
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Task] Task ended during gripper rotation!")
            return

        # --- Step 3: Move to anchor position (move) ---
        if drawer_anchor_pos is not None:
            print(f"[Task] Moving to anchor position of {drawer_name} drawer.")
            obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
            if done:
                print("[Task] Task ended during move to drawer anchor position!")
                return
        else:
            print("[Error] Drawer anchor position not found!")
            return

        # --- Step 4: Pick the drawer handle (pick) ---
        # Use anchor position as the target for picking the handle
        print(f"[Task] Picking the {drawer_name} drawer handle.")
        obs, reward, done = pick(env, task, target_pos=drawer_anchor_pos, approach_distance=0.10, approach_axis='y')
        if done:
            print("[Task] Task ended during picking drawer handle!")
            return

        # --- Step 5: Pull the drawer open (pull) ---
        # Pull along the x axis (assume positive x is outwards), distance ~0.18m
        print(f"[Task] Pulling the {drawer_name} drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.18, pull_axis='x')
        if done:
            print("[Task] Task ended during pulling drawer!")
            return

        # --- Step 6: Move to a safe waypoint before picking tomatoes (move) ---
        if waypoint1 is not None:
            print("[Task] Moving to safe waypoint before picking tomatoes.")
            obs, reward, done = move(env, task, target_pos=waypoint1)
            if done:
                print("[Task] Task ended during move to waypoint!")
                return

        # --- Step 7: Rotate gripper back to 0 degrees (rotate) ---
        # Quaternion for 0 deg about z: [0, 0, 0, 1]
        target_quat_zero = np.array([0, 0, 0, 1])
        print("[Task] Rotating gripper back to 0 degrees.")
        obs, reward, done = rotate(env, task, target_quat=target_quat_zero)
        if done:
            print("[Task] Task ended during gripper rotation to 0 deg!")
            return

        # --- Step 8: Move above first tomato (move) ---
        if item1_pos is not None:
            print("[Task] Moving above item1 (tomato1).")
            obs, reward, done = move(env, task, target_pos=item1_pos + np.array([0, 0, 0.10]))
            if done:
                print("[Task] Task ended during move above item1!")
                return
        else:
            print("[Error] item1 position not found!")
            return

        # --- Step 9: Pick item1 (pick) ---
        print("[Task] Picking item1 (tomato1).")
        obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during picking item1!")
            return

        # --- Step 10: Place item1 on plate (place) ---
        if plate_pos is not None:
            print("[Task] Placing item1 on plate.")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during placing item1!")
                return
        else:
            print("[Error] Plate position not found!")
            return

        # --- Step 11: Move above item2 (move) ---
        if item2_pos is not None:
            print("[Task] Moving above item2 (tomato2).")
            obs, reward, done = move(env, task, target_pos=item2_pos + np.array([0, 0, 0.10]))
            if done:
                print("[Task] Task ended during move above item2!")
                return
        else:
            print("[Error] item2 position not found!")
            return

        # --- Step 12: Pick item2 (pick) ---
        print("[Task] Picking item2 (tomato2).")
        obs, reward, done = pick(env, task, target_pos=item2_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during picking item2!")
            return

        # --- Step 13: Place item2 on plate (place) ---
        print("[Task] Placing item2 on plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during placing item2!")
            return

        # --- (Optional) Repeat for item3 if needed ---
        if item3_pos is not None:
            print("[Task] Moving above item3 (tomato3).")
            obs, reward, done = move(env, task, target_pos=item3_pos + np.array([0, 0, 0.10]))
            if done:
                print("[Task] Task ended during move above item3!")
                return

            print("[Task] Picking item3 (tomato3).")
            obs, reward, done = pick(env, task, target_pos=item3_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during picking item3!")
                return

            print("[Task] Placing item3 on plate.")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during placing item3!")
                return

        print("[Task] All tomatoes placed on plate. Task completed successfully!")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()