import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all the tomatoes and leave them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation:
        # Drawers: 'bottom', 'middle', 'top'
        # Drawer positions: 'bottom_anchor_pos', 'bottom_side_pos', etc.
        # Items: 'item1', 'item2', 'item3' (assume these are the tomatoes)
        # Plate: 'plate'

        # --- Step 1: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] 'bottom_side_pos' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to bottom drawer side position.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended during move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for grasping drawer handle) ---
        # Get the current gripper position and set target orientation to 90 deg about z
        # We'll use the skill's convention: quaternion for 90 deg about z axis
        from scipy.spatial.transform import Rotation as R
        quat_ninety_deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        # Ensure it's np.array and normalized
        quat_ninety_deg = np.array(quat_ninety_deg) / np.linalg.norm(quat_ninety_deg)
        print("[Plan] Rotating gripper to 90 degrees.")
        obs, reward, done = rotate(env, task, target_quat=quat_ninety_deg)
        if done:
            print("[Task] Task ended during rotate to 90 deg!")
            return

        # --- Step 3: Move to anchor position of the bottom drawer (handle) ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] 'bottom_anchor_pos' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to bottom drawer anchor position.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended during move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick the drawer handle (simulate grasp) ---
        # Use pick skill to close gripper on the anchor position
        print("[Plan] Grasping the bottom drawer handle.")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during pick (drawer handle)!")
            return

        # --- Step 5: Pull the drawer open (fully) ---
        # Pull along the x axis (assume drawer opens along +x, adjust if needed)
        # Use a reasonable pull distance (e.g., 0.20 meters)
        pull_distance = 0.20
        print("[Plan] Pulling the bottom drawer open.")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended during pull (open drawer)!")
            return

        # --- Step 6: Move gripper away from drawer to avoid collision (back to side pos) ---
        print("[Plan] Moving away from drawer (to side position).")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended during move away from drawer!")
            return

        # --- Step 7: Rotate gripper back to 0 degrees (for picking objects) ---
        quat_zero_deg = R.from_euler('z', 0, degrees=True).as_quat()
        quat_zero_deg = np.array(quat_zero_deg) / np.linalg.norm(quat_zero_deg)
        print("[Plan] Rotating gripper to 0 degrees.")
        obs, reward, done = rotate(env, task, target_quat=quat_zero_deg)
        if done:
            print("[Task] Task ended during rotate to 0 deg!")
            return

        # --- Step 8: Move to first tomato (item1) ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] 'item1' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to item1 (tomato1).")
        obs, reward, done = move(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Task ended during move to item1!")
            return

        # --- Step 9: Pick item1 (tomato1) ---
        print("[Plan] Picking up item1 (tomato1).")
        obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during pick (item1)!")
            return

        # --- Step 10: Place item1 on the plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] 'plate' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Placing item1 on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during place (item1)!")
            return

        # --- Step 11: Move to item2 (tomato2) ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] 'item2' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to item2 (tomato2).")
        obs, reward, done = move(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Task ended during move to item2!")
            return

        # --- Step 12: Pick item2 (tomato2) ---
        print("[Plan] Picking up item2 (tomato2).")
        obs, reward, done = pick(env, task, target_pos=item2_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during pick (item2)!")
            return

        # --- Step 13: Place item2 on the plate ---
        print("[Plan] Placing item2 on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during place (item2)!")
            return

        # --- (Optional) Repeat for item3 if present ---
        if 'item3' in positions:
            print("[Plan] Moving to item3 (tomato3).")
            obs, reward, done = move(env, task, target_pos=positions['item3'])
            if done:
                print("[Task] Task ended during move to item3!")
                return

            print("[Plan] Picking up item3 (tomato3).")
            obs, reward, done = pick(env, task, target_pos=positions['item3'], approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during pick (item3)!")
                return

            print("[Plan] Placing item3 on the plate.")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during place (item3)!")
                return

        print("[Task] All tomatoes placed on the plate. Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()