import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Unlock a cabinet by opening an accessible drawer, then transfer the 2 tomatoes onto the plate, leaving any other items undisturbed.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object name mapping (from object list and observation)
        # item1, item2: tomatoes; item3: rubbish; plate: destination
        # Drawer positions: bottom_anchor_pos, bottom_side_pos, etc.

        # --- Step 1: Move to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] bottom_side_pos not found in positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 1: Move to bottom drawer side position.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer handle approach) ---
        # We'll use a quaternion for 90 degrees rotation about z axis
        # Assume initial orientation is aligned with zero_deg (identity quaternion)
        # We'll rotate about z by 90 degrees
        from scipy.spatial.transform import Rotation as R
        quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        print("[Plan] Step 2: Rotate gripper to 90 degrees.")
        obs, reward, done = rotate(env, task, target_quat=quat_90deg)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 3: Pull the drawer open ---
        # Move to anchor position first (simulate grasping the handle)
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] bottom_anchor_pos not found in positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 3a: Move to bottom drawer anchor position.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # Pull: pull along x axis (assume drawer opens along +x, pull distance ~0.15m)
        print("[Plan] Step 3b: Pull the drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("[Task] Task ended after pull!")
            return

        # --- Step 4: Move above the first tomato (item1) ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] item1 not found in positions.")
            shutdown_environment(env)
            return

        # Approach from above (z axis)
        item1_above = np.array(item1_pos) + np.array([0, 0, 0.10])
        print("[Plan] Step 4: Move above item1 (tomato1).")
        obs, reward, done = move(env, task, target_pos=item1_above)
        if done:
            print("[Task] Task ended after move above item1!")
            return

        # --- Step 5: Pick up the first tomato (item1) ---
        print("[Plan] Step 5: Pick up item1 (tomato1).")
        obs, reward, done = pick(env, task, target_pos=np.array(item1_pos), approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # --- Step 6: Move above the plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] plate not found in positions.")
            shutdown_environment(env)
            return

        plate_above = np.array(plate_pos) + np.array([0, 0, 0.10])
        print("[Plan] Step 6: Move above plate.")
        obs, reward, done = move(env, task, target_pos=plate_above)
        if done:
            print("[Task] Task ended after move above plate!")
            return

        # --- Step 7: Place the first tomato (item1) on the plate ---
        print("[Plan] Step 7: Place item1 (tomato1) on plate.")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after place item1!")
            return

        # --- Step 8: Move above the second tomato (item2) ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] item2 not found in positions.")
            shutdown_environment(env)
            return

        item2_above = np.array(item2_pos) + np.array([0, 0, 0.10])
        print("[Plan] Step 8: Move above item2 (tomato2).")
        obs, reward, done = move(env, task, target_pos=item2_above)
        if done:
            print("[Task] Task ended after move above item2!")
            return

        # --- Step 9: Pick up the second tomato (item2) ---
        print("[Plan] Step 9: Pick up item2 (tomato2).")
        obs, reward, done = pick(env, task, target_pos=np.array(item2_pos), approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # --- Step 10: Move above the plate again ---
        print("[Plan] Step 10: Move above plate (for item2).")
        obs, reward, done = move(env, task, target_pos=plate_above)
        if done:
            print("[Task] Task ended after move above plate (item2)!")
            return

        # --- Step 11: Place the second tomato (item2) on the plate ---
        print("[Plan] Step 11: Place item2 (tomato2) on plate.")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after place item2!")
            return

        print("[Task] Task completed successfully! Both tomatoes placed on plate and drawer opened.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()