import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions: dict mapping object names to np.array([x, y, z]) positions

        # --- Plan Execution (Oracle Plan) ---
        # Step 1: Rotate gripper to 90 degrees (to prepare for side approach)
        try:
            # Get current gripper pose
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # Target: 90 deg about z axis (assuming 'zero_deg' is [0,0,0,1], 'ninety_deg' is [0,0,sin(45°),cos(45°)])
            # We'll use scipy to get the quaternion for 90 deg about z
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('z', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move gripper to side position of an accessible drawer (bottom drawer)
        try:
            # Use 'bottom_side_pos' as the side position for the bottom drawer
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                side_pos = positions.get('side-pos-bottom', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to side position: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move gripper to anchor position of the same drawer (bottom drawer)
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                anchor_pos = positions.get('anchor-pos-bottom', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to anchor position: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick the drawer handle (simulate pick-drawer by using pick at anchor position)
        try:
            # For the drawer, we simulate the pick by closing the gripper at the anchor position
            # Use pick with a small approach distance and approach_axis='z'
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.05, approach_axis='z')
            if done:
                print("[Task] Task ended during pick drawer!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick drawer: {e}")
            shutdown_environment(env)
            return

        # Step 5: Pull the drawer open (pull along x axis, positive direction)
        try:
            # Estimate a reasonable pull distance (e.g., 0.15m)
            pull_distance = 0.15
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick up tomato1 from table
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in object positions.")
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            return

        # Step 7: Place tomato1 on plate
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during place tomato1: {e}")
            shutdown_environment(env)
            return

        # Step 8: Pick up tomato2 from table
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in object positions.")
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # Step 9: Place tomato2 on plate
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()