# (No code block, plain text as requested)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer, then toss the 2 tomatoes from the table onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos', 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'

        # --- Step 1: Rotate gripper to 90 degrees (for drawer manipulation) ---
        # We'll use the bottom drawer (unlocked) for simplicity.
        # Get current gripper pose
        obs = task.get_observation()
        start_quat = obs.gripper_pose[3:7]
        # Target: 90 deg about z axis (assuming 'zero_deg' is [0,0,0,1], 'ninety_deg' is a 90deg rotation)
        # We'll use scipy to compute the quaternion for 90deg about z
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        try:
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] rotate failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move to side position of bottom drawer ---
        try:
            side_pos = positions['bottom_side_pos']
            obs, reward, done = move(env, task, target_pos=side_pos)
            if done:
                print("[Task] Task ended after move to side pos!")
                return
        except Exception as e:
            print(f"[ERROR] move to side pos failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            anchor_pos = positions['bottom_anchor_pos']
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor pos!")
                return
        except Exception as e:
            print(f"[ERROR] move to anchor pos failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: (Pick drawer handle) ---
        # The plan would require a "pick-drawer" skill, but only 'pick' is available.
        # We'll skip explicit pick-drawer and assume the gripper is at the anchor and ready to pull.

        # --- Step 5: Pull the drawer open ---
        # Pull along x axis (assume positive x opens the drawer), distance ~0.15m
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] pull failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up tomato1 from table ---
        try:
            tomato1_pos = positions['tomato1']
            obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] pick tomato1 failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place tomato1 on plate ---
        try:
            plate_pos = positions['plate']
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] place tomato1 failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 8: Pick up tomato2 from table ---
        try:
            tomato2_pos = positions['tomato2']
            obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] pick tomato2 failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 on plate ---
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] place tomato2 failed: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
