import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their 3D positions
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Plan steps (from specification):
        # 1. rotate gripper to ninety_deg
        # 2. move to side-pos-bottom
        # 3. move to anchor-pos-bottom
        # 4. pick bottom drawer at anchor-pos-bottom
        # 5. pull bottom drawer
        # 6. pick tomato1 from table
        # 7. place tomato1 on plate
        # 8. pick tomato2 from table
        # 9. place tomato2 on plate

        # --- Step 1: Rotate gripper to ninety_deg ---
        # Assume we have a mapping from angle name to quaternion
        # For this environment, let's define the mapping here:
        angle_quat_map = {
            'zero_deg': np.array([0, 0, 0, 1]),  # identity quaternion
            'ninety_deg': np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]),  # 90 deg about z
        }
        try:
            target_quat = angle_quat_map['ninety_deg']
        except Exception as e:
            print("[Error] Could not find quaternion for ninety_deg:", e)
            shutdown_environment(env)
            return

        print("[Task] Step 1: Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 2: Move to side-pos-bottom ---
        # Use the position for 'bottom_side_pos'
        try:
            target_pos = positions['bottom_side_pos']
        except Exception as e:
            print("[Error] Could not find position for bottom_side_pos:", e)
            shutdown_environment(env)
            return

        print("[Task] Step 2: Moving to bottom_side_pos")
        obs, reward, done = move(env, task, target_pos)
        if done:
            print("[Task] Task ended after move to side-pos-bottom!")
            return

        # --- Step 3: Move to anchor-pos-bottom ---
        try:
            target_pos = positions['bottom_anchor_pos']
        except Exception as e:
            print("[Error] Could not find position for bottom_anchor_pos:", e)
            shutdown_environment(env)
            return

        print("[Task] Step 3: Moving to bottom_anchor_pos")
        obs, reward, done = move(env, task, target_pos)
        if done:
            print("[Task] Task ended after move to anchor-pos-bottom!")
            return

        # --- Step 4: Pick bottom drawer at anchor-pos-bottom ---
        # For picking the drawer, use the anchor position as the target
        print("[Task] Step 4: Picking bottom drawer at anchor-pos-bottom")
        obs, reward, done = pick(env, task, target_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after picking bottom drawer!")
            return

        # --- Step 5: Pull bottom drawer ---
        # Pull along the x-axis (assume positive x is outwards)
        # Use a reasonable pull distance (e.g., 0.15m)
        pull_distance = 0.15
        print("[Task] Step 5: Pulling bottom drawer")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended after pulling bottom drawer!")
            return

        # --- Step 6: Pick tomato1 from table ---
        try:
            tomato1_pos = positions['tomato1']
        except Exception as e:
            print("[Error] Could not find position for tomato1:", e)
            shutdown_environment(env)
            return

        print("[Task] Step 6: Picking tomato1 from table")
        obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking tomato1!")
            return

        # --- Step 7: Place tomato1 on plate ---
        try:
            plate_pos = positions['plate']
        except Exception as e:
            print("[Error] Could not find position for plate:", e)
            shutdown_environment(env)
            return

        print("[Task] Step 7: Placing tomato1 on plate")
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # --- Step 8: Pick tomato2 from table ---
        try:
            tomato2_pos = positions['tomato2']
        except Exception as e:
            print("[Error] Could not find position for tomato2:", e)
            shutdown_environment(env)
            return

        print("[Task] Step 8: Picking tomato2 from table")
        obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking tomato2!")
            return

        # --- Step 9: Place tomato2 on plate ---
        print("[Task] Step 9: Placing tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()