import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their 3D positions (np.ndarray)
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Unlock the cabinet by opening an accessible drawer, then transfer the 2 tomatoes onto the plate, leaving any other items undisturbed.

        # Step 1: Rotate gripper to 90 degrees (for side approach to drawer)
        # We'll use the 'rotate' skill to set the gripper orientation to 90 degrees about z.
        # Assume 'ninety_deg' means a rotation of 90 degrees about the relevant axis.
        # We'll use the current gripper position and set the orientation accordingly.
        try:
            print("[Step 1] Rotating gripper to 90 degrees for drawer approach.")
            # Get current gripper pose
            obs = task.get_observation()
            # We'll rotate about z axis by 90 degrees
            # RLBench uses xyzw quaternion; 90 deg about z: [0, 0, sqrt(0.5), sqrt(0.5)]
            target_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        try:
            print("[Step 2] Moving gripper to side position of bottom drawer.")
            # Use 'bottom_side_pos' as the approach position for the bottom drawer
            if 'bottom_side_pos' in positions:
                side_pos = positions['bottom_side_pos']
            elif 'side-pos-bottom' in positions:
                side_pos = positions['side-pos-bottom']
            else:
                raise KeyError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended after move to side position!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to side) failed: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move gripper to anchor position of the bottom drawer (for grasping handle)
        try:
            print("[Step 3] Moving gripper to anchor position of bottom drawer.")
            if 'bottom_anchor_pos' in positions:
                anchor_pos = positions['bottom_anchor_pos']
            elif 'anchor-pos-bottom' in positions:
                anchor_pos = positions['anchor-pos-bottom']
            else:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor position!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to anchor) failed: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick the drawer handle (simulate grasping the drawer)
        try:
            print("[Step 4] Picking (grasping) the bottom drawer handle.")
            # For pick, we use the anchor position as the target
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after picking drawer handle!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (pick drawer handle) failed: {e}")
            shutdown_environment(env)
            return

        # Step 5: Pull the drawer open
        try:
            print("[Step 5] Pulling the bottom drawer open.")
            # Pull along the x axis (assume drawer opens along +x)
            # We'll use a reasonable pull distance (e.g., 0.15 meters)
            pull_distance = 0.15
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pulling drawer!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull) failed: {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick up tomato1 from the table
        try:
            print("[Step 6] Picking up tomato1 from table.")
            if 'tomato1' not in positions:
                raise KeyError("tomato1 not found in object positions.")
            tomato1_pos = positions['tomato1']
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after picking tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # Step 7: Place tomato1 on the plate
        try:
            print("[Step 7] Placing tomato1 on plate.")
            if 'plate' not in positions:
                raise KeyError("plate not found in object positions.")
            plate_pos = positions['plate']
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after placing tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 7 (place tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # Step 8: Pick up tomato2 from the table
        try:
            print("[Step 8] Picking up tomato2 from table.")
            if 'tomato2' not in positions:
                raise KeyError("tomato2 not found in object positions.")
            tomato2_pos = positions['tomato2']
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after picking tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 8 (pick tomato2) failed: {e}")
            shutdown_environment(env)
            return

        # Step 9: Place tomato2 on the plate
        try:
            print("[Step 9] Placing tomato2 on plate.")
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after placing tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 9 (place tomato2) failed: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Both tomatoes placed on plate and drawer opened.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()