import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their 3D positions
        positions = get_object_positions()

        # Defensive: Check required objects exist
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos',
            'middle_side_pos', 'middle_anchor_pos', 'middle_joint_pos',
            'top_side_pos', 'top_anchor_pos', 'top_joint_pos',
            'item1', 'item2', 'item3', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                print(f"[ERROR] Required object '{k}' not found in object_positions.")
                shutdown_environment(env)
                return

        # --- PLAN EXECUTION (according to the provided plan and goal) ---

        # [Frozen Code Start]
        obs, reward, done = move(
                env,
                task,
                target_pos=positions['bottom_side_pos'],
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer handle) ---
        print("[Plan] Step 2: Rotate gripper to 90 degrees")
        from scipy.spatial.transform import Rotation as R
        obs_now = task.get_observation()
        current_quat = obs_now.gripper_pose[3:7]
        r = R.from_quat(current_quat)
        r_90 = R.from_euler('z', 90, degrees=True) * r
        target_quat_90 = r_90.as_quat()
        obs, reward, done = rotate(
            env,
            task,
            target_quat=target_quat_90,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate to 90 deg!")
            return

        # --- Step 3: Move to anchor position of the bottom drawer (handle) ---
        print("[Plan] Step 3: Move to bottom_anchor_pos")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['bottom_anchor_pos'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick the drawer handle (simulate grasp) ---
        print("[Plan] Step 4: Pick the bottom drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=positions['bottom_anchor_pos'],
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after picking drawer handle!")
            return

        # --- Step 5: Pull the drawer open ---
        print("[Plan] Step 5: Pull the bottom drawer open")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.15,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # --- Step 6: Move to a neutral position above the table (for safety) ---
        print("[Plan] Step 6: Move to neutral position above table")
        table_z = positions['item1'][2]
        neutral_pos = np.array([positions['item1'][0], positions['item1'][1], table_z + 0.15])
        obs, reward, done = move(
            env,
            task,
            target_pos=neutral_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after moving to neutral position!")
            return

        # --- Step 7: Rotate gripper back to 0 degrees (for picking tomatoes) ---
        print("[Plan] Step 7: Rotate gripper to 0 degrees")
        obs_now = task.get_observation()
        current_quat = obs_now.gripper_pose[3:7]
        r = R.from_quat(current_quat)
        r_0 = R.from_euler('z', 0, degrees=True)
        target_quat_0 = r_0.as_quat()
        obs, reward, done = rotate(
            env,
            task,
            target_quat=target_quat_0,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate to 0 deg!")
            return
        # [Frozen Code End]

        # --- Step 8: Pick tomato1 from table ---
        print("[Plan] Step 8: Pick tomato1 from table")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=positions['item1'],
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after picking tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick of tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato1 on plate ---
        print("[Plan] Step 9: Place tomato1 on plate")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=positions['plate'],
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after placing tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during place of tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 10: Pick tomato2 from table ---
        print("[Plan] Step 10: Pick tomato2 from table")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=positions['item2'],
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after picking tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick of tomato2: {e}")
            shutdown_environment(env)
            return

        # --- Step 11: Place tomato2 on plate ---
        print("[Plan] Step 11: Place tomato2 on plate")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=positions['plate'],
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after placing tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during place of tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Plan completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[ERROR] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()