import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all the tomatoes and leave them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'plate'
        # ]
        # We'll treat item1, item2, item3 as the tomatoes.

        # Defensive: check all required keys exist
        required_keys = [
            'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
            'item1', 'item2', 'item3', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Missing object position for: {k}")

        # Choose the 'bottom' drawer for opening
        drawer_anchor_pos = positions['bottom_anchor_pos']
        drawer_side_pos = positions['bottom_side_pos']
        drawer_joint_pos = positions['bottom_joint_pos']
        plate_pos = positions['plate']
        tomato_names = ['item1', 'item2', 'item3']
        tomato_positions = [positions[name] for name in tomato_names]

        # Helper: quaternion for 90 deg about z
        def quat_about_z(deg):
            rad = np.deg2rad(deg)
            return np.array([0, 0, np.sin(rad/2), np.cos(rad/2)])

        # [Frozen Code Start]
        target_quat_90 = quat_about_z(90)
        obs, reward, done = rotate(
            env, task, target_quat=target_quat_90
        )
        obs, reward, done = move(
            env, task, target_pos=drawer_side_pos
        )
        obs, reward, done = move(
            env, task, target_pos=drawer_anchor_pos
        )
        # [Frozen Code End]

        if done:
            print("[Task] Task ended after initial steps!")
            return

        # Step 4: pick-drawer (simulate grasping the drawer at anchor)
        print("[Step 4] Pick (grasp) the drawer at anchor position")
        # In this environment, we simulate the drawer grasp by moving to the anchor position (already done)
        # If a specific pick-drawer skill is needed, it would be called here.
        # For now, proceed to move to the joint position for pulling.

        obs, reward, done = move(
            env, task, target_pos=drawer_joint_pos
        )
        if done:
            print("[Task] Task ended after move to joint pos!")
            return

        # Step 5: pull to open the drawer
        print("[Step 5] Pull to open the drawer")
        # Pull along x axis (assume drawer opens along +x, adjust if needed)
        pull_distance = 0.18  # Adjust as needed for full open
        try:
            obs, reward, done = pull(
                env, task, pull_distance=pull_distance, pull_axis='x'
            )
        except Exception as e:
            print("[Task] Exception during pull:", e)
            return
        if done:
            print("[Task] Task ended after pull!")
            return

        # Step 6: pick first tomato (item1)
        print("[Step 6] Pick first tomato (item1)")
        try:
            obs, reward, done = pick(
                env, task, target_pos=positions['item1'], approach_distance=0.08, approach_axis='-z'
            )
        except Exception as e:
            print("[Task] Exception during pick item1:", e)
            return
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # Step 7: place first tomato on plate
        print("[Step 7] Place first tomato on plate")
        try:
            obs, reward, done = place(
                env, task, target_pos=plate_pos, approach_distance=0.08, approach_axis='-z'
            )
        except Exception as e:
            print("[Task] Exception during place item1:", e)
            return
        if done:
            print("[Task] Task ended after place item1!")
            return

        # Step 8: pick second tomato (item2)
        print("[Step 8] Pick second tomato (item2)")
        try:
            obs, reward, done = pick(
                env, task, target_pos=positions['item2'], approach_distance=0.08, approach_axis='-z'
            )
        except Exception as e:
            print("[Task] Exception during pick item2:", e)
            return
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # Step 9: place second tomato on plate
        print("[Step 9] Place second tomato on plate")
        try:
            obs, reward, done = place(
                env, task, target_pos=plate_pos, approach_distance=0.08, approach_axis='-z'
            )
        except Exception as e:
            print("[Task] Exception during place item2:", e)
            return
        if done:
            print("[Task] Task ended after place item2!")
            return

        # Step 10: pick third tomato (item3)
        print("[Step 10] Pick third tomato (item3)")
        try:
            obs, reward, done = pick(
                env, task, target_pos=positions['item3'], approach_distance=0.08, approach_axis='-z'
            )
        except Exception as e:
            print("[Task] Exception during pick item3:", e)
            return
        if done:
            print("[Task] Task ended after pick item3!")
            return

        # Step 11: place third tomato on plate
        print("[Step 11] Place third tomato on plate")
        try:
            obs, reward, done = place(
                env, task, target_pos=plate_pos, approach_distance=0.08, approach_axis='-z'
            )
        except Exception as e:
            print("[Task] Exception during place item3:", e)
            return
        if done:
            print("[Task] Task ended after place item3!")
            return

        print("[Task] Finished plan steps from specification.")

    except Exception as e:
        print("[Task] Exception occurred:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()