import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Slide open any available drawer to access its contents, then toss the 2 tomatoes from the table onto the plate.
        # We will:
        # 1. Move to the side position of a drawer (e.g., bottom drawer)
        # 2. Rotate gripper to 90 degrees (for grasping drawer handle)
        # 3. Move to anchor position (handle)
        # 4. Pick the drawer handle
        # 5. Pull the drawer open
        # 6. Move away from the drawer
        # 7. Rotate gripper back to 0 degrees (for object pick)
        # 8. Move to above item1 (tomato1)
        # 9. Pick item1
        # 10. Move to above plate
        # 11. Place item1 on plate
        # 12. Move to above item2 (tomato2)
        # 13. Pick item2
        # 14. Move to above plate
        # 15. Place item2 on plate

        # --- Step 1: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos', None)
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            print("[Step 1] Move to bottom_side_pos:", bottom_side_pos)
        except Exception as e:
            print(f"[Error] Step 1 (move to bottom_side_pos) failed: {e}")
            return

        # [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        # [Frozen Code End]
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer handle) ---
        try:
            from scipy.spatial.transform import Rotation as R
            quat_90 = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            print("[Step 2] Rotate gripper to 90 degrees (quaternion):", quat_90)
            obs, reward, done = rotate(env, task, target_quat=quat_90)
            if done:
                print("[Task] Task ended after rotate to 90 deg!")
                return
        except Exception as e:
            print(f"[Error] Step 2 (rotate to 90 deg) failed: {e}")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            print("[Step 3] Move to bottom_anchor_pos:", bottom_anchor_pos)
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Error] Step 3 (move to bottom_anchor_pos) failed: {e}")
            return

        # --- Step 4: Pick the drawer handle (simulate pick-drawer) ---
        try:
            print("[Step 4] Pick the drawer handle at anchor position.")
            obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after picking drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Step 4 (pick drawer handle) failed: {e}")
            return

        # --- Step 5: Pull the drawer open ---
        try:
            print("[Step 5] Pull the drawer open along x axis.")
            obs, reward, done = pull(env, task, pull_distance=0.18, pull_axis='x')
            if done:
                print("[Task] Task ended after pulling drawer!")
                return
        except Exception as e:
            print(f"[Error] Step 5 (pull drawer) failed: {e}")
            return

        # --- Step 6: Move away from drawer (to side position) ---
        try:
            print("[Step 6] Move back to bottom_side_pos:", bottom_side_pos)
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            if done:
                print("[Task] Task ended after moving away from drawer!")
                return
        except Exception as e:
            print(f"[Error] Step 6 (move away from drawer) failed: {e}")
            return

        # --- Step 7: Rotate gripper back to 0 degrees (for object pick) ---
        try:
            quat_0 = R.from_euler('z', 0, degrees=True).as_quat()
            print("[Step 7] Rotate gripper to 0 degrees (quaternion):", quat_0)
            obs, reward, done = rotate(env, task, target_quat=quat_0)
            if done:
                print("[Task] Task ended after rotate to 0 deg!")
                return
        except Exception as e:
            print(f"[Error] Step 7 (rotate to 0 deg) failed: {e}")
            return

        # --- Step 8: Move to above item1 (tomato1) ---
        try:
            item1_pos = positions.get('item1', None)
            if item1_pos is None:
                raise KeyError("item1 not found in positions.")
            item1_above = np.array(item1_pos) + np.array([0, 0, 0.10])
            print("[Step 8] Move to above item1:", item1_above)
            obs, reward, done = move(env, task, target_pos=item1_above)
            if done:
                print("[Task] Task ended after moving above item1!")
                return
        except Exception as e:
            print(f"[Error] Step 8 (move above item1) failed: {e}")
            return

        # --- Step 9: Pick item1 (tomato1) ---
        try:
            print("[Step 9] Pick item1 at:", item1_pos)
            obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after picking item1!")
                return
        except Exception as e:
            print(f"[Error] Step 9 (pick item1) failed: {e}")
            return

        # --- Step 10: Move to above plate ---
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in positions.")
            plate_above = np.array(plate_pos) + np.array([0, 0, 0.10])
            print("[Step 10] Move to above plate:", plate_above)
            obs, reward, done = move(env, task, target_pos=plate_above)
            if done:
                print("[Task] Task ended after moving above plate!")
                return
        except Exception as e:
            print(f"[Error] Step 10 (move above plate) failed: {e}")
            return

        # --- Step 11: Place item1 on plate ---
        try:
            print("[Step 11] Place item1 on plate at:", plate_pos)
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after placing item1!")
                return
        except Exception as e:
            print(f"[Error] Step 11 (place item1) failed: {e}")
            return

        # --- Step 12: Move to above item2 (tomato2) ---
        try:
            item2_pos = positions.get('item2', None)
            if item2_pos is None:
                raise KeyError("item2 not found in positions.")
            item2_above = np.array(item2_pos) + np.array([0, 0, 0.10])
            print("[Step 12] Move to above item2:", item2_above)
            obs, reward, done = move(env, task, target_pos=item2_above)
            if done:
                print("[Task] Task ended after moving above item2!")
                return
        except Exception as e:
            print(f"[Error] Step 12 (move above item2) failed: {e}")
            return

        # --- Step 13: Pick item2 (tomato2) ---
        try:
            print("[Step 13] Pick item2 at:", item2_pos)
            obs, reward, done = pick(env, task, target_pos=item2_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after picking item2!")
                return
        except Exception as e:
            print(f"[Error] Step 13 (pick item2) failed: {e}")
            return

        # --- Step 14: Move to above plate ---
        try:
            print("[Step 14] Move to above plate:", plate_above)
            obs, reward, done = move(env, task, target_pos=plate_above)
            if done:
                print("[Task] Task ended after moving above plate (2nd time)!")
                return
        except Exception as e:
            print(f"[Error] Step 14 (move above plate) failed: {e}")
            return

        # --- Step 15: Place item2 on plate ---
        try:
            print("[Step 15] Place item2 on plate at:", plate_pos)
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after placing item2!")
                return
        except Exception as e:
            print(f"[Error] Step 15 (place item2) failed: {e}")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()