import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer to unlock the cabinet, then transfer the 2 tomatoes onto the plate, leaving other items undisturbed.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'plate'
        # ]
        # We assume item1 and item2 are the tomatoes, item3 is "rubbish" (should not be moved).
        # The plate is the goal location for the tomatoes.

        # Drawer opening: We'll use the "bottom" drawer for accessibility.
        # We'll use the following positions for the drawer:
        #   - bottom_side_pos: side position to approach
        #   - bottom_anchor_pos: anchor position to grip
        #   - bottom_joint_pos: position to pull along (for pull action)
        # For gripper orientation, we assume:
        #   - zero_deg: default orientation (quaternion)
        #   - ninety_deg: rotated orientation (quaternion)
        # We'll use the skill_code's move/rotate/pick/pull/place as provided.

        # Helper: get quaternion for zero_deg and ninety_deg
        # We'll use RLBench's convention: [x, y, z, w] (xyzw)
        # For zero_deg: no rotation (identity quaternion)
        zero_deg_quat = np.array([0, 0, 0, 1])
        # For ninety_deg: rotate 90 deg about z axis
        ninety_deg_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # Get all needed positions
        try:
            bottom_side_pos = np.array(positions['bottom_side_pos'])
            bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
            bottom_joint_pos = np.array(positions['bottom_joint_pos'])
            item1_pos = np.array(positions['item1'])
            item2_pos = np.array(positions['item2'])
            plate_pos = np.array(positions['plate'])
        except Exception as e:
            print("[Error] Could not retrieve required object positions:", e)
            shutdown_environment(env)
            return

        # === PLAN EXECUTION ===
        # [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        obs, reward, done = pull(env, task, pull_distance=0.18, pull_axis='x')
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = rotate(env, task, target_quat=zero_deg_quat)
        obs, reward, done = move(env, task, target_pos=item1_pos)
        obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.10, approach_axis='z')
        # [Frozen Code End]

        # === Continue Plan: Place item1 on plate, then move and place item2 ===

        if done:
            print("[Task] Task ended after picking item1!")
            return

        print("[Step 10] Place item1 on plate")
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
        except Exception as e:
            print("[Error] Exception during place of item1:", e)
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after placing item1!")
            return

        print("[Step 11] Move to item2 (second tomato)")
        try:
            obs, reward, done = move(env, task, target_pos=item2_pos)
        except Exception as e:
            print("[Error] Exception during move to item2:", e)
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after moving to item2!")
            return

        print("[Step 12] Pick item2 (second tomato)")
        try:
            obs, reward, done = pick(env, task, target_pos=item2_pos, approach_distance=0.10, approach_axis='z')
        except Exception as e:
            print("[Error] Exception during pick of item2:", e)
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after picking item2!")
            return

        print("[Step 13] Place item2 on plate")
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
        except Exception as e:
            print("[Error] Exception during place of item2:", e)
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after placing item2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()