import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then toss the 2 tomatoes from the table onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys expected in positions:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'

        # === 9-Step Plan Execution (Strictly as per Specification) ===

        # Step 1: rotate (rotate gripper to 90 degrees for drawer grasp)
        try:
            # 90 deg about z-axis: quaternion [0, 0, sin(π/4), cos(π/4)]
            ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
            print("[Step 1] Rotating gripper to 90 degrees for drawer grasp.")
            [Frozen Code Start]
    obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
[Frozen Code End]
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Step 1 (rotate) failed: {e}")
            return

        # Step 2: move (move to side position of bottom drawer)
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            print("[Step 2] Moving to bottom drawer side position.")
            [Frozen Code Start]
    obs, reward, done = move(
                env,
                task,
                target_pos=bottom_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
[Frozen Code End]
            if done:
                print("[Task] Task ended during move to side pos!")
                return
        except Exception as e:
            print(f"[Error] Step 2 (move to side pos) failed: {e}")
            return

        # Step 3: move (move to anchor position of bottom drawer)
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            print("[Step 3] Moving to bottom drawer anchor position.")
            [Frozen Code Start]
    obs, reward, done = move(
                env,
                task,
                target_pos=bottom_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
[Frozen Code End]
            if done:
                print("[Task] Task ended during move to anchor pos!")
                return
        except Exception as e:
            print(f"[Error] Step 3 (move to anchor pos) failed: {e}")
            return

        # Step 4: pick (grasp the drawer handle at anchor position)
        try:
            print("[Step 4] Grasping the bottom drawer handle.")
            [Frozen Code Start]
    obs, reward, done = pick(
                env,
                task,
                target_pos=bottom_anchor_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='y',  # Assume approach from front
                timeout=10.0
            )
[Frozen Code End]
            if done:
                print("[Task] Task ended during pick drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Step 4 (pick drawer handle) failed: {e}")
            return

        # Step 5: pull (pull the drawer open along x-axis)
        try:
            pull_distance = 0.18  # Typical drawer pull distance
            print("[Step 5] Pulling the bottom drawer open.")
            [Frozen Code Start]
    obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
[Frozen Code End]
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Error] Step 5 (pull drawer) failed: {e}")
            return

        # Step 6: pick (pick up tomato1 from table)
        try:
            tomato1_pos = positions.get('tomato1')
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in positions.")
            print("[Step 6] Picking up tomato1 from table.")
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.12,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick tomato1!")
                return
        except Exception as e:
            print(f"[Error] Step 6 (pick tomato1) failed: {e}")
            return

        # Step 7: place (place tomato1 on plate)
        try:
            plate_pos = positions.get('plate')
            if plate_pos is None:
                raise KeyError("plate not found in positions.")
            print("[Step 7] Placing tomato1 on plate.")
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.12,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place tomato1!")
                return
        except Exception as e:
            print(f"[Error] Step 7 (place tomato1) failed: {e}")
            return

        # Step 8: pick (pick up tomato2 from table)
        try:
            tomato2_pos = positions.get('tomato2')
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in positions.")
            print("[Step 8] Picking up tomato2 from table.")
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato2_pos,
                approach_distance=0.12,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick tomato2!")
                return
        except Exception as e:
            print(f"[Error] Step 8 (pick tomato2) failed: {e}")
            return

        # Step 9: place (place tomato2 on plate)
        try:
            print("[Step 9] Placing tomato2 on plate.")
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.12,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place tomato2!")
                return
        except Exception as e:
            print(f"[Error] Step 9 (place tomato2) failed: {e}")
            return

        print("[Task] Task completed successfully! Both tomatoes placed on plate after opening drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()