import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one drawer, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object name mapping (from object list and plan)
        # Provided objects: bottom_anchor_pos, bottom_joint_pos, bottom_side_pos, middle_anchor_pos, middle_joint_pos, middle_side_pos, top_anchor_pos, top_joint_pos, top_side_pos, waypoint1, rubbish, tomato1, tomato2, plate

        # Helper to get object positions robustly
        def safe_get_pos(name, fallback=None):
            if name in positions:
                return positions[name]
            alt = name.replace('-', '_')
            if alt in positions:
                return positions[alt]
            if fallback is not None:
                return fallback
            raise KeyError(f"Object position for '{name}' not found in positions dict.")

        # Step 1: rotate(gripper, zero_deg, ninety_deg)
        # We'll assume the gripper's orientation for zero_deg and ninety_deg are provided in positions or can be constructed
        # If not, we can use a default orientation and rotate 90 deg about z axis

        # Get current gripper orientation (as quaternion)
        obs = task.get_observation()
        gripper_quat = obs.gripper_pose[3:7]
        from scipy.spatial.transform import Rotation as R
        zero_deg_quat = gripper_quat
        ninety_deg_quat = (R.from_quat(zero_deg_quat) * R.from_euler('z', 90, degrees=True)).as_quat()

        # [Frozen Code Start]
        obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        # [Frozen Code End]

        # Step 2: move(gripper, side-pos-bottom)
        try:
            print("[Task] Step 2: Moving gripper to side-pos-bottom")
            target_pos = safe_get_pos('bottom_side_pos')
            # [Frozen Code Start]
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(target_pos)
            # [Frozen Code End]
            , max_steps=100, threshold=0.01, timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to side-pos-bottom!")
                return
        except Exception as e:
            print(f"[Error] Step 2 (move to side-pos-bottom) failed: {e}")
            shutdown_environment(env)
            return

        # Step 3: move(gripper, anchor-pos-bottom)
        try:
            print("[Task] Step 3: Moving gripper to anchor-pos-bottom")
            target_pos = safe_get_pos('bottom_anchor_pos')
            # [Frozen Code Start]
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(target_pos)
            # [Frozen Code End]
            , max_steps=100, threshold=0.01, timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to anchor-pos-bottom!")
                return
        except Exception as e:
            print(f"[Error] Step 3 (move to anchor-pos-bottom) failed: {e}")
            shutdown_environment(env)
            return

        # Step 4: pick(bottom, anchor-pos-bottom)
        # Here, 'bottom' is the drawer, and we want to pick it at anchor-pos-bottom
        try:
            print("[Task] Step 4: Picking bottom drawer at anchor-pos-bottom")
            target_pos = safe_get_pos('bottom_anchor_pos')
            # [Frozen Code Start]
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(target_pos)
            # [Frozen Code End]
            , approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick (drawer)!")
                return
        except Exception as e:
            print(f"[Error] Step 4 (pick drawer) failed: {e}")
            shutdown_environment(env)
            return

        # Step 5: pull(gripper, bottom)
        # For pull, we need to specify pull_distance and axis.
        # Let's assume the drawer opens along the x-axis, and a reasonable pull distance (e.g., 0.15m)
        try:
            print("[Task] Step 5: Pulling bottom drawer")
            pull_distance = 0.15
            pull_axis = 'x'
            # [Frozen Code Start]
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis=pull_axis,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            # [Frozen Code End]
            if done:
                print("[Task] Task ended during pull (drawer)!")
                return
        except Exception as e:
            print(f"[Error] Step 5 (pull drawer) failed: {e}")
            shutdown_environment(env)
            return

        # Step 6: pick(tomato1, table)
        try:
            print("[Task] Step 6: Picking tomato1 from table")
            tomato1_pos = safe_get_pos('tomato1')
            # [Frozen Code Start]
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(tomato1_pos)
            # [Frozen Code End]
            , approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick (tomato1)!")
                return
        except Exception as e:
            print(f"[Error] Step 6 (pick tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # Step 7: place(tomato1, plate)
        try:
            print("[Task] Step 7: Placing tomato1 on plate")
            plate_pos = safe_get_pos('plate')
            obs, reward, done = place(
                env,
                task,
                target_pos=np.array(plate_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place (tomato1)!")
                return
        except Exception as e:
            print(f"[Error] Step 7 (place tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # Step 8: pick(tomato2, table)
        try:
            print("[Task] Step 8: Picking tomato2 from table")
            tomato2_pos = safe_get_pos('tomato2')
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(tomato2_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick (tomato2)!")
                return
        except Exception as e:
            print(f"[Error] Step 8 (pick tomato2) failed: {e}")
            shutdown_environment(env)
            return

        # Step 9: place(tomato2, plate)
        try:
            print("[Task] Step 9: Placing tomato2 on plate")
            plate_pos = safe_get_pos('plate')
            obs, reward, done = place(
                env,
                task,
                target_pos=np.array(plate_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place (tomato2)!")
                return
        except Exception as e:
            print(f"[Error] Step 9 (place tomato2) failed: {e}")
            shutdown_environment(env)
            return

        print("[Task] All steps completed successfully! Task should be complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()