import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any unlocked drawer, then place both tomatoes onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'tomato1', 'tomato2', 'plate', 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # --- Step 1: Rotate gripper to 90 degrees (vertical for drawer manipulation) ---
        # We'll use the 'ninety_deg' orientation for the gripper.
        # Assume the target orientation is provided as a quaternion in positions or can be constructed.
        # If not, we can use a standard quaternion for 90deg rotation about x (or y) axis.
        # Let's try to get the current gripper pose and rotate about x by 90deg.
        obs = task.get_observation()
        start_quat = obs.gripper_pose[3:7]
        from scipy.spatial.transform import Rotation as R
        # Compose a 90deg rotation about x axis
        ninety_deg_rot = R.from_euler('x', 90, degrees=True)
        start_rot = R.from_quat(start_quat)
        target_rot = ninety_deg_rot * start_rot
        target_quat = target_rot.as_quat()
        try:
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move gripper to the side position of a drawer (let's use bottom drawer) ---
        # Use 'bottom_side_pos' as the approach position for the gripper
        try:
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side position: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move gripper to the anchor position of the drawer (for grasping handle) ---
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor position: {e}")
            shutdown_environment(env)
            return

        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        obs, reward, done = move(env, task, anchor_pos)
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        obs, reward, done = move(env, task, anchor_pos)
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        obs, reward, done = move(env, task, anchor_pos)
        obs, reward, done = move(env, task, joint_pos)
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        obs, reward, done = pick(env, task, tomato1_pos)
        obs, reward, done = place(env, task, plate_pos)
        # [Frozen Code End]

        # --- Step 8: Pick up tomato2 from the table ---
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in object positions.")
            obs, reward, done = pick(env, task, tomato2_pos)
            if done:
                print("[Task] Task ended during pick tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 on the plate ---
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            obs, reward, done = place(env, task, plate_pos)
            if done:
                print("[Task] Task ended during place tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()