import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===
        # Step 1: Rotate gripper to 90 degrees (assume 'ninety_deg_quat' is provided by the environment or object_positions)
        print("[Plan Step 1] Rotating gripper to 90 degrees for drawer manipulation")
        if 'ninety_deg_quat' in positions:
            ninety_deg_quat = positions['ninety_deg_quat']
        else:
            # Fallback: try to get from env/task or define manually
            # Here, we assume the drawer requires a rotation about z by 90 degrees
            from scipy.spatial.transform import Rotation as R
            ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended during rotate!")
            shutdown_environment(env)
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        print("[Plan Step 2] Moving gripper to bottom_side_pos")
        if 'bottom_side_pos' not in positions:
            print("[Error] 'bottom_side_pos' not found in positions.")
            shutdown_environment(env)
            return
        target_pos = positions['bottom_side_pos']
        obs, reward, done = move(env, task, target_pos=target_pos)
        if done:
            print("[Task] Task ended during move to bottom_side_pos!")
            shutdown_environment(env)
            return

        # Step 3: Move gripper to the anchor position of the bottom drawer
        print("[Plan Step 3] Moving gripper to bottom_anchor_pos")
        if 'bottom_anchor_pos' not in positions:
            print("[Error] 'bottom_anchor_pos' not found in positions.")
            shutdown_environment(env)
            return
        target_pos = positions['bottom_anchor_pos']
        obs, reward, done = move(env, task, target_pos=target_pos)
        if done:
            print("[Task] Task ended during move to bottom_anchor_pos!")
            shutdown_environment(env)
            return

        # Step 4: (pick-drawer) is not available as a skill, so we skip this step.

        # Step 5: Pull the drawer open (assume along +x axis, 0.18m)
        print("[Plan Step 5] Pulling the bottom drawer open")
        pull_distance = 0.18
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended during pull!")
            shutdown_environment(env)
            return

        # Step 6: Pick up tomato1 from the table
        print("[Plan Step 6] Picking up tomato1")
        if 'tomato1' not in positions:
            print("[Error] 'tomato1' not found in positions.")
            shutdown_environment(env)
            return
        tomato1_pos = positions['tomato1']
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Task ended during pick tomato1!")
            shutdown_environment(env)
            return

        # Step 7: Place tomato1 on the plate
        print("[Plan Step 7] Placing tomato1 on plate")
        if 'plate' not in positions:
            print("[Error] 'plate' not found in positions.")
            shutdown_environment(env)
            return
        plate_pos = positions['plate']
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended during place tomato1!")
            shutdown_environment(env)
            return

        # Step 8: Pick up tomato2 from the table
        print("[Plan Step 8] Picking up tomato2")
        if 'tomato2' not in positions:
            print("[Error] 'tomato2' not found in positions.")
            shutdown_environment(env)
            return
        tomato2_pos = positions['tomato2']
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Task ended during pick tomato2!")
            shutdown_environment(env)
            return

        # Step 9: Place tomato2 on the plate
        print("[Plan Step 9] Placing tomato2 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended during place tomato2!")
            shutdown_environment(env)
            return

        print("[Task] All plan steps executed. Task completed successfully!")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
        shutdown_environment(env)
        return

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()