import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names in the environment
        # 'item1', 'item2', 'item3' are the tomatoes
        # 'plate' is the destination
        # Drawer anchor/side/joint positions: e.g., 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # For robustness, check all required keys
        required_keys = [
            'item1', 'item2', 'item3', 'plate',
            'bottom_anchor_pos', 'bottom_side_pos', 'bottom_joint_pos',
            'middle_anchor_pos', 'middle_side_pos', 'middle_joint_pos',
            'top_anchor_pos', 'top_side_pos', 'top_joint_pos'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Missing object position for: {k}")

        # Choose a drawer to open (bottom drawer for simplicity)
        drawer_anchor = positions['bottom_anchor_pos']
        drawer_side = positions['bottom_side_pos']
        drawer_joint = positions['bottom_joint_pos']

        # Plate position
        plate_pos = positions['plate']

        # Tomato positions
        tomato_names = ['item1', 'item2', 'item3']
        tomato_positions = [positions[name] for name in tomato_names]

        # Step 1: Move to side of bottom drawer
        print("[Step 1] Move to side of bottom drawer")
        obs, reward, done = move(env, task, target_pos=drawer_side)
        if done:
            print("[Task] Task ended after move to drawer side!")
            return

        # Step 2: Rotate gripper to 90 deg (vertical)
        print("[Step 2] Rotate gripper to 90 deg")
        # Assume 90 deg is [x, y, z, w] quaternion for vertical orientation
        # If object_positions provides 'ninety_deg_quat', use it; else, use a default
        ninety_deg_quat = None
        if 'ninety_deg_quat' in positions:
            ninety_deg_quat = positions['ninety_deg_quat']
        else:
            from scipy.spatial.transform import Rotation as R
            ninety_deg_quat = R.from_euler('x', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to anchor position (drawer handle)
        print("[Step 3] Move to anchor position (drawer handle)")
        obs, reward, done = move(env, task, target_pos=drawer_anchor)
        if done:
            print("[Task] Task ended after move to anchor!")
            return

        # Step 4: Pick (grasp) drawer handle
        print("[Step 4] Pick (grasp) drawer handle")
        # Use pick skill to close gripper at anchor position
        obs, reward, done = pick(env, task, target_pos=drawer_anchor, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after pick drawer handle!")
            return

        # Step 5: Pull to open drawer
        print("[Step 5] Pull to open drawer")
        # Pull along x axis (assume drawer opens along +x, adjust if needed)
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("[Task] Task ended after pull!")
            return

        # Step 6: Move away from drawer (to side position)
        print("[Step 6] Move away from drawer (side pos)")
        obs, reward, done = move(env, task, target_pos=drawer_side)
        if done:
            print("[Task] Task ended after move away from drawer!")
            return

        # Step 7: Rotate gripper back to 0 deg (horizontal)
        print("[Step 7] Rotate gripper to 0 deg")
        zero_deg_quat = None
        if 'zero_deg_quat' in positions:
            zero_deg_quat = positions['zero_deg_quat']
        else:
            from scipy.spatial.transform import Rotation as R
            zero_deg_quat = R.from_euler('x', 0, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=zero_deg_quat)
        if done:
            print("[Task] Task ended after rotate to 0 deg!")
            return

        # For each tomato: move, pick, move to plate, place
        for idx, (tomato_name, tomato_pos) in enumerate(zip(tomato_names, tomato_positions)):
            print(f"[Step {8+3*idx}] Move to {tomato_name}")
            obs, reward, done = move(env, task, target_pos=tomato_pos)
            if done:
                print(f"[Task] Task ended after move to {tomato_name}!")
                return

            print(f"[Step {9+3*idx}] Pick {tomato_name}")
            obs, reward, done = pick(env, task, target_pos=tomato_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print(f"[Task] Task ended after pick {tomato_name}!")
                return

            print(f"[Step {10+3*idx}] Move to plate")
            obs, reward, done = move(env, task, target_pos=plate_pos)
            if done:
                print(f"[Task] Task ended after move to plate!")
                return

            print(f"[Step {11+3*idx}] Place {tomato_name} on plate")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print(f"[Task] Task ended after place {tomato_name}!")
                return

        print("[Task] All tomatoes placed on plate. Task completed successfully!")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()