import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any unlocked drawer, then drop the 2 tomatoes onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object name mapping ---
        # From object list and observation, the two tomatoes are 'item1' and 'item2'
        # The plate is 'plate'
        # Drawers: 'bottom', 'middle', 'top' (but only 'bottom' is not locked in the initial state)
        # Drawer handle/anchor positions: 'bottom_anchor_pos', 'middle_anchor_pos', 'top_anchor_pos'
        # Drawer side positions: 'bottom_side_pos', etc.

        # 1. Find an unlocked drawer (not locked in initial state)
        # According to the observation, 'top' and 'middle' are locked, so 'bottom' is not locked.
        # We'll use 'bottom' drawer.

        # 2. Get the anchor and side positions for the 'bottom' drawer
        try:
            anchor_pos = positions['bottom_anchor_pos']
            side_pos = positions['bottom_side_pos']
        except KeyError as e:
            print(f"[Error] Could not find required drawer positions: {e}")
            shutdown_environment(env)
            return

        # 3. Get the gripper's current position (should be 'nowhere_pos' at start)
        # We'll move to the side position first, then to the anchor, then pull.

        # 4. Get the two tomato positions and the plate position
        try:
            tomato1_pos = positions['item1']
            tomato2_pos = positions['item2']
            plate_pos = positions['plate']
        except KeyError as e:
            print(f"[Error] Could not find required object positions: {e}")
            shutdown_environment(env)
            return

        # === Step 1: Pull open any unlocked drawer ===
        # Plan:
        #   a) Move gripper to side position of bottom drawer (side_pos)
        #   b) Rotate gripper to 90 degrees (if needed)
        #   c) Move gripper to anchor position (anchor_pos)
        #   d) Pick the drawer handle (simulate grasp)
        #   e) Pull the drawer open

        # a) Move to side position
        print("[Task] Moving gripper to bottom drawer side position:", side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=side_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended unexpectedly after moving to side position!")
            return

        # b) Rotate gripper to 90 degrees (z axis)
        # We'll assume 90 degrees about z is [0, 0, sin(45°), cos(45°)] in quaternion (xyzw)
        # But let's get the current orientation and rotate about z by 90 deg
        from scipy.spatial.transform import Rotation as R
        current_quat = obs.gripper_pose[3:7]
        rot = R.from_quat(current_quat)
        rot_90 = R.from_euler('z', 90, degrees=True)
        target_quat = (rot * rot_90).as_quat()
        print("[Task] Rotating gripper to 90 degrees about z axis")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended unexpectedly after rotating gripper!")
            return

        # c) Move to anchor position
        print("[Task] Moving gripper to bottom drawer anchor position:", anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=anchor_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended unexpectedly after moving to anchor position!")
            return

        # d) Pick the drawer handle (simulate grasp)
        print("[Task] Grasping the drawer handle at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended unexpectedly after picking drawer handle!")
            return

        # e) Pull the drawer open (along x axis, positive direction)
        print("[Task] Pulling the drawer open")
        # Pull distance: estimate, e.g., 0.15 meters
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.15,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended unexpectedly after pulling drawer!")
            return

        # === Step 2: Drop the 2 tomatoes onto the plate ===
        # For each tomato: pick from table, place on plate

        # --- Tomato 1 ---
        print("[Task] Picking up tomato1 (item1) at:", tomato1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended unexpectedly after picking tomato1!")
            return

        print("[Task] Placing tomato1 on plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after placing tomato1! Reward:", reward)
            return

        # --- Tomato 2 ---
        print("[Task] Picking up tomato2 (item2) at:", tomato2_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended unexpectedly after picking tomato2!")
            return

        print("[Task] Placing tomato2 on plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed after placing tomato2! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()