import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all the tomatoes and leave them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation:
        # Drawers: 'bottom', 'middle', 'top'
        # Items: 'item1', 'item2', 'item3'
        # Plate: 'plate'
        # Drawer positions: e.g., 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # For this plan, select the 'bottom' drawer (arbitrary choice as per goal)
        drawer = 'bottom'
        # Drawer positions
        anchor_pos_name = 'bottom_anchor_pos'
        side_pos_name = 'bottom_side_pos'
        # Plate position
        plate_pos = positions.get('plate')
        # Tomato/item positions (from object list, items are 'item1', 'item2', 'item3')
        tomato_names = ['item1', 'item2', 'item3']
        tomato_positions = [positions.get(name) for name in tomato_names]

        # Defensive: check all required positions exist
        anchor_pos = positions.get(anchor_pos_name)
        side_pos = positions.get(side_pos_name)
        if not (anchor_pos is not None and side_pos is not None and plate_pos is not None):
            print("[Error] One or more required positions for the drawer or plate are missing.")
            return
        for idx, pos in enumerate(tomato_positions):
            if pos is None:
                print(f"[Error] Position for tomato {tomato_names[idx]} is missing.")
                return

        # === Plan Execution ===
        # Plan steps (from specification):
        # 1. rotate (rotate gripper to 90 degrees for drawer grasp)
        print("[Step 1] Rotate gripper to 90 degrees (vertical).")
        # Assume 'ninety_deg' is a quaternion in positions or can be constructed
        # If not available, rotate around x by 90 deg (RLBench uses xyzw)
        ninety_deg_quat = np.array([0.7071, 0, 0, 0.7071])  # x-axis 90deg
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # 2. move (move gripper to drawer side position)
        print("[Step 2] Move gripper to drawer side position.")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Task ended after move to side position!")
            return

        # 3. move (move gripper to anchor position for drawer grasp)
        print("[Step 3] Move gripper to drawer anchor position.")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Task ended after move to anchor position!")
            return

        # 4. pick (grasp the drawer handle at anchor position)
        print("[Step 4] Pick (grasp) the drawer handle at anchor position.")
        # For drawer, use pick with anchor_pos (simulate grasping handle)
        obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after picking drawer handle!")
            return

        # 5. pull (pull the drawer open along x axis)
        print("[Step 5] Pull the drawer open (along x axis).")
        # Assume positive x is the pull direction; pull distance is estimated (e.g., 0.18m)
        obs, reward, done = pull(env, task, pull_distance=0.18, pull_axis='x')
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # For each tomato: pick and place on plate
        for idx, tomato_pos in enumerate(tomato_positions):
            print(f"[Step 6.{idx+1}] Move gripper above tomato {tomato_names[idx]}.")
            obs, reward, done = move(env, task, target_pos=tomato_pos)
            if done:
                print(f"[Task] Task ended after moving to {tomato_names[idx]}!")
                return
            print(f"[Step 7.{idx+1}] Pick up {tomato_names[idx]}.")
            obs, reward, done = pick(env, task, target_pos=tomato_pos, approach_distance=0.12, approach_axis='z')
            if done:
                print(f"[Task] Task ended after picking {tomato_names[idx]}!")
                return
            print(f"[Step 8.{idx+1}] Place {tomato_names[idx]} on plate.")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.12, approach_axis='z')
            if done:
                print(f"[Task] Task ended after placing {tomato_names[idx]}!")
                return

        print("[Task] Task completed successfully! Reward:", reward)

    except Exception as e:
        print("[Error] Exception occurred during task execution:", str(e))
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()