import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Task: Unlock a cabinet by opening an accessible drawer, then transfer the 2 tomatoes onto the plate, leaving any other items undisturbed.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation:
        # 'item1', 'item2', 'item3' (assume item1 and item2 are tomatoes, item3 is rubbish)
        # 'plate'
        # Drawer positions: 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # Defensive: check all required keys exist
        required_keys = [
            'item1', 'item2', 'item3', 'plate',
            'bottom_anchor_pos', 'bottom_side_pos', 'bottom_joint_pos',
            'middle_anchor_pos', 'middle_side_pos', 'middle_joint_pos',
            'top_anchor_pos', 'top_side_pos', 'top_joint_pos',
            'waypoint1'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Missing object position for: {k}")

        # === Step 1: Move to bottom drawer side position ===
        print("[Plan] Step 1: Move to bottom drawer side position")
        obs, reward, done = move(
            env, task,
            target_pos=positions['bottom_side_pos'],
            max_steps=100, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Task ended at step 1!")
            return

        # === Step 2: Rotate gripper to 90 degrees (for drawer handle) ===
        print("[Plan] Step 2: Rotate gripper to 90 degrees")
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        obs, reward, done = rotate(
            env, task,
            target_quat=ninety_deg_quat,
            max_steps=100, threshold=0.05, timeout=10.0
        )
        if done:
            print("[Task] Task ended at step 2!")
            return

        # === Step 3: Move to bottom drawer anchor position (handle) ===
        print("[Plan] Step 3: Move to bottom drawer anchor position")
        obs, reward, done = move(
            env, task,
            target_pos=positions['bottom_anchor_pos'],
            max_steps=100, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Task ended at step 3!")
            return

        # === Step 4: Pick the drawer handle (simulate grasp) ===
        print("[Plan] Step 4: Pick the bottom drawer handle")
        # For pick, approach_axis is usually '-y' or '-z' depending on drawer orientation.
        # We'll use approach_axis='y' (assuming handle is in front of gripper)
        try:
            obs, reward, done = pick(
                env, task,
                target_pos=positions['bottom_anchor_pos'],
                approach_distance=0.10,
                max_steps=100, threshold=0.01, approach_axis='y', timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during pick (drawer handle): {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 4!")
            return

        # === Step 5: Pull the drawer open ===
        print("[Plan] Step 5: Pull the bottom drawer open")
        # Pull along the x axis (assume drawer opens along +x, adjust if needed)
        try:
            obs, reward, done = pull(
                env, task,
                pull_distance=0.18,  # typical drawer pull distance
                pull_axis='x',
                max_steps=100, threshold=0.01, timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during pull (drawer): {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 5!")
            return

        # === Step 6: Move to a safe waypoint above the table ===
        print("[Plan] Step 6: Move to waypoint1 (safe above table)")
        try:
            obs, reward, done = move(
                env, task,
                target_pos=positions['waypoint1'],
                max_steps=100, threshold=0.01, timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during move to waypoint1: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 6!")
            return

        # === Step 7: Rotate gripper back to 0 degrees (for picking tomatoes) ===
        print("[Plan] Step 7: Rotate gripper to 0 degrees")
        zero_deg_quat = R.from_euler('z', 0, degrees=True).as_quat()
        try:
            obs, reward, done = rotate(
                env, task,
                target_quat=zero_deg_quat,
                max_steps=100, threshold=0.05, timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during rotate to 0 deg: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 7!")
            return

        # === Step 8: Pick up item1 (tomato1) from table ===
        print("[Plan] Step 8: Pick up item1 (tomato1) from table")
        try:
            obs, reward, done = pick(
                env, task,
                target_pos=positions['item1'],
                approach_distance=0.15,
                max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during pick (item1): {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 8!")
            return

        # === Step 9: Place item1 (tomato1) on plate ===
        print("[Plan] Step 9: Place item1 (tomato1) on plate")
        try:
            obs, reward, done = place(
                env, task,
                target_pos=positions['plate'],
                approach_distance=0.15,
                max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during place (item1): {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 9!")
            return

        # === Step 10: Move to waypoint1 (safe above table) ===
        print("[Plan] Step 10: Move to waypoint1 (safe above table)")
        try:
            obs, reward, done = move(
                env, task,
                target_pos=positions['waypoint1'],
                max_steps=100, threshold=0.01, timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during move to waypoint1 (after item1): {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 10!")
            return

        # === Step 11: Pick up item2 (tomato2) from table ===
        print("[Plan] Step 11: Pick up item2 (tomato2) from table")
        try:
            obs, reward, done = pick(
                env, task,
                target_pos=positions['item2'],
                approach_distance=0.15,
                max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during pick (item2): {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 11!")
            return

        # === Step 12: Place item2 (tomato2) on plate ===
        print("[Plan] Step 12: Place item2 (tomato2) on plate")
        try:
            obs, reward, done = place(
                env, task,
                target_pos=positions['plate'],
                approach_distance=0.15,
                max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during place (item2): {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended at step 12!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()