import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then toss the 2 tomatoes from the table onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation:
        # Drawers: 'bottom', 'middle', 'top'
        # Items: 'item1', 'item2', 'item3' (assume item1 and item2 are tomatoes)
        # Locations: 'table', 'plate'
        # Drawer positions: e.g., 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # --- Step 1: Move to side position of an available drawer (choose 'bottom') ---
        # We'll use 'bottom_side_pos' as the approach position for the gripper.
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_side_pos' in object positions.")
            shutdown_environment(env)
            return

        [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = rotate(env, task, target_quat=quat_ninety_z)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        obs, reward, done = move(env, task, target_pos=item1_pos)
        [Frozen Code End]
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # --- Step 6: (Optional) Rotate gripper to default orientation if needed ---
        from scipy.spatial.transform import Rotation as R
        quat_zero = R.from_euler('z', 0, degrees=True).as_quat()
        print("[Plan] Rotating gripper to 0 degrees (default) for pick.")
        obs, reward, done = rotate(env, task, target_quat=quat_zero)
        if done:
            print("[Task] Task ended after rotate to default!")
            return

        # --- Step 7: Pick up tomato1 (item1) from table ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] Could not find 'item1' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Picking up item1 (tomato1).")
        obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking item1!")
            return

        # --- Step 8: Place tomato1 onto plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] Could not find 'plate' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Placing item1 (tomato1) onto plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing item1!")
            return

        # --- Step 9: Move to item2 (tomato2) position ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] Could not find 'item2' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to item2 (tomato2) position.")
        obs, reward, done = move(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Task ended after move to item2!")
            return

        # --- Step 10: Pick up tomato2 (item2) from table ---
        print("[Plan] Picking up item2 (tomato2).")
        obs, reward, done = pick(env, task, target_pos=item2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking item2!")
            return

        # --- Step 11: Place tomato2 onto plate ---
        print("[Plan] Placing item2 (tomato2) onto plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing item2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()