import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any unlocked drawer, then place both tomatoes onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Defensive: check all required objects exist
        required_keys = [
            'bottom_anchor_pos', 'bottom_side_pos', 'bottom_joint_pos',
            'middle_anchor_pos', 'middle_side_pos', 'middle_joint_pos',
            'top_anchor_pos', 'top_side_pos', 'top_joint_pos',
            'tomato1', 'tomato2', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                print(f"[Error] Required object '{k}' not found in positions dictionary.")
                shutdown_environment(env)
                return

        # --- Step 1: Rotate gripper to 90 degrees (for side approach) ---
        obs = task.get_observation()
        ninety_deg_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_joint_pos = positions['bottom_joint_pos']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_joint_pos)
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
        # [Frozen Code End]

        if done:
            print("[Task] Task ended during frozen code steps!")
            return

        # --- Step 7: Place tomato1 on plate ---
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during place tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 8: Pick up tomato2 from table ---
        try:
            obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 on plate ---
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()