import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Slide open any available drawer to access its contents, then toss the 2 tomatoes from the table onto the plate.

        # --- Step 1: Rotate gripper to 90 degrees (for side approach to drawer) ---
        # We'll use the bottom drawer (unlocked) for simplicity.
        # Find the quaternion for 90 degrees rotation about z axis.
        from scipy.spatial.transform import Rotation as R
        ninety_deg_euler = [0, 0, np.pi/2]  # 90 deg about z
        target_quat = R.from_euler('xyz', ninety_deg_euler).as_quat()  # [x, y, z, w]
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 2: Move to side position of bottom drawer ---
        try:
            side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[ERROR] bottom_side_pos not found in positions.")
            return
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Task ended after move to side!")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[ERROR] bottom_anchor_pos not found in positions.")
            return
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Task ended after move to anchor!")
            return

        # --- Step 4: Pick the drawer handle (simulate grasp) ---
        # For the drawer, we use the anchor position as the pick target
        try:
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick drawer!")
                return
        except Exception as e:
            print(f"[ERROR] pick drawer failed: {e}")
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along the x axis (assume positive x is outwards), distance empirically set
        try:
            pull_distance = 0.18  # meters, adjust as needed for your environment
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] pull drawer failed: {e}")
            return

        # --- Step 6: Pick up tomato1 from the table ---
        try:
            tomato1_pos = positions['tomato1']
        except KeyError:
            print("[ERROR] tomato1 not found in positions.")
            return
        try:
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] pick tomato1 failed: {e}")
            return

        # --- Step 7: Place tomato1 onto the plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[ERROR] plate not found in positions.")
            return
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] place tomato1 failed: {e}")
            return

        # --- Step 8: Pick up tomato2 from the table ---
        try:
            tomato2_pos = positions['tomato2']
        except KeyError:
            print("[ERROR] tomato2 not found in positions.")
            return
        try:
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] pick tomato2 failed: {e}")
            return

        # --- Step 9: Place tomato2 onto the plate ---
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] place tomato2 failed: {e}")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()