# run_skeleton_task.py (Completed with Exploration Phase and Robustness)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Consistency Check: Object Declarations ===
        # Ensure all objects in code are present in positions
        # (If not, print a warning)
        required_objects = ['item1', 'item2', 'item3', 'plate']
        for obj in required_objects:
            if obj not in positions:
                print(f"[Warning] Object '{obj}' not found in object_positions. Please check object list consistency.")

        # === Safety Check: Gripper State ===
        try:
            obs = task.get_observation()
            gripper_state = getattr(obs, 'gripper_openness', None)
            if gripper_state is not None:
                print(f"[Safety] Gripper openness at start: {gripper_state}")
            else:
                print("[Safety] Gripper openness attribute not found in observation.")
        except Exception as e:
            print(f"[Error] Could not check gripper state: {e}")

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to explore the environment to discover which predicate is missing.
        # We will attempt to use the available skills to interact with all objects and locations,
        # and observe if any expected state change or feedback is missing.

        # For demonstration, we will:
        # 1. Move to each object's location (if move is available)
        # 2. Attempt to pick each object (if pick is available)
        # 3. Attempt to place each object (if place is available)
        # 4. Attempt to rotate and pull if applicable

        # Available skills: ['pick', 'place', 'move', 'rotate', 'pull']
        # We will use try-except to handle errors gracefully.

        # Track which predicates we can confirm through actions
        discovered_predicates = set()
        max_attempts = 5
        attempt = 0

        for obj in required_objects:
            if obj not in positions:
                print(f"[Exploration] Skipping '{obj}' as it is not in positions.")
                continue
            obj_pos = positions[obj]
            print(f"[Exploration] Attempting to interact with '{obj}' at position {obj_pos}")

            # 1. Move to object's location (if move is available)
            try:
                print(f"[Exploration] Moving to '{obj}'...")
                obs, reward, done = move(env, task, target_pos=obj_pos, max_steps=100, threshold=0.01, timeout=10.0)
                discovered_predicates.add('robot-at')
                print(f"[Exploration] Moved to '{obj}'.")
                if done:
                    print("[Exploration] Task ended after move!")
                    break
            except Exception as e:
                print(f"[Error] Move to '{obj}' failed: {e}")

            # 2. Try to pick the object
            try:
                print(f"[Exploration] Picking '{obj}'...")
                obs, reward, done = pick(env, task, target_pos=obj_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                discovered_predicates.add('holding')
                discovered_predicates.add('weight-known')  # As per exploration domain, pick reveals weight
                discovered_predicates.add('durability-known')  # As per exploration domain, pick reveals durability
                print(f"[Exploration] Picked '{obj}'.")
                if done:
                    print("[Exploration] Task ended after pick!")
                    break
            except Exception as e:
                print(f"[Error] Pick '{obj}' failed: {e}")

            # 3. Try to place the object (if place is available)
            try:
                # For demonstration, place at a default location (e.g., 'plate' if not the same object)
                if 'plate' in positions and obj != 'plate':
                    place_pos = positions['plate']
                    print(f"[Exploration] Placing '{obj}' on 'plate' at {place_pos}...")
                    obs, reward, done = place(env, task, target_pos=place_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                    discovered_predicates.add('at')
                    print(f"[Exploration] Placed '{obj}' on 'plate'.")
                    if done:
                        print("[Exploration] Task ended after place!")
                        break
            except Exception as e:
                print(f"[Error] Place '{obj}' failed: {e}")

            # 4. Try to rotate (if rotate is available and object is rotatable)
            try:
                if 'rotate' in globals():
                    # For demonstration, try rotating to a default quaternion (identity)
                    print(f"[Exploration] Rotating gripper for '{obj}'...")
                    target_quat = np.array([0, 0, 0, 1])
                    obs, reward, done = rotate(env, task, target_quat, max_steps=50, threshold=0.05, timeout=5.0)
                    discovered_predicates.add('rotated')
                    print(f"[Exploration] Rotated gripper for '{obj}'.")
                    if done:
                        print("[Exploration] Task ended after rotate!")
                        break
            except Exception as e:
                print(f"[Error] Rotate for '{obj}' failed: {e}")

            # 5. Try to pull (if pull is available and object is pullable)
            try:
                if 'pull' in globals():
                    print(f"[Exploration] Pulling '{obj}'...")
                    obs, reward, done = pull(env, task, obj)
                    discovered_predicates.add('lock-known')
                    print(f"[Exploration] Pulled '{obj}'.")
                    if done:
                        print("[Exploration] Task ended after pull!")
                        break
            except Exception as e:
                print(f"[Error] Pull for '{obj}' failed: {e}")

            attempt += 1
            if attempt >= max_attempts:
                print("[Exploration] Reached maximum exploration attempts.")
                break

        # === Exploration Summary ===
        print(f"[Exploration] Discovered predicates through actions: {discovered_predicates}")

        # === Task Completion Logic ===
        # After exploration, check if the task is completed (e.g., all objects placed on plate)
        # This is a placeholder; actual completion logic depends on the specific task.
        try:
            obs = task.get_observation()
            # Example: check if all objects are at the plate position
            all_on_plate = True
            for obj in required_objects:
                if obj == 'plate' or obj not in positions:
                    continue
                # Here, you would check the actual state (e.g., via obs or environment API)
                # For demonstration, we assume success if no exception occurred
            if all_on_plate:
                print("[Task] All objects placed on plate (assumed). Task complete.")
        except Exception as e:
            print(f"[Error] Task completion check failed: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()