# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try All Available Skills to Detect Missing Predicate ===
        # The goal is to execute each available skill in a safe, exploratory way to see if any predicate is missing.
        # This helps identify which predicate is not being updated or is missing in the domain/problem.

        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']
        exploration_results = {}

        # For demonstration, we will attempt to call each skill with plausible arguments.
        # If a skill fails due to a missing predicate or precondition, we log the error.

        # 1. Try 'move' skill (if implemented in skill_code)
        if 'move' in available_skills:
            try:
                # Try moving between two locations if available
                location_keys = [k for k in positions.keys() if 'location' in k or 'room' in k or 'pos' in k]
                if len(location_keys) >= 2:
                    from_loc = location_keys[0]
                    to_loc = location_keys[1]
                    print(f"[Exploration] Trying move from {from_loc} to {to_loc}")
                    obs, reward, done = move(env, task, from_loc, to_loc)
                    exploration_results['move'] = 'success'
                else:
                    print("[Exploration] Not enough locations to test move.")
                    exploration_results['move'] = 'skipped'
            except Exception as e:
                print(f"[Exploration] move failed: {e}")
                exploration_results['move'] = f'failed: {e}'

        # 2. Try 'pick' skill
        if 'pick' in available_skills:
            try:
                # Find an object and its location
                object_keys = [k for k in positions.keys() if 'object' in k or 'drawer' in k or 'ball' in k or 'item' in k]
                location_keys = [k for k in positions.keys() if 'location' in k or 'room' in k or 'pos' in k]
                if object_keys and location_keys:
                    obj = object_keys[0]
                    loc = location_keys[0]
                    print(f"[Exploration] Trying pick on {obj} at {loc}")
                    obs, reward, done = pick(env, task, obj, loc)
                    exploration_results['pick'] = 'success'
                else:
                    print("[Exploration] Not enough objects/locations to test pick.")
                    exploration_results['pick'] = 'skipped'
            except Exception as e:
                print(f"[Exploration] pick failed: {e}")
                exploration_results['pick'] = f'failed: {e}'

        # 3. Try 'place' skill
        if 'place' in available_skills:
            try:
                # Place the object picked above to another location if possible
                if object_keys and len(location_keys) > 1:
                    obj = object_keys[0]
                    target_loc = location_keys[1]
                    print(f"[Exploration] Trying place of {obj} at {target_loc}")
                    obs, reward, done = place(env, task, obj, target_loc)
                    exploration_results['place'] = 'success'
                else:
                    print("[Exploration] Not enough objects/locations to test place.")
                    exploration_results['place'] = 'skipped'
            except Exception as e:
                print(f"[Exploration] place failed: {e}")
                exploration_results['place'] = f'failed: {e}'

        # 4. Try 'rotate' skill
        if 'rotate' in available_skills:
            try:
                # Try rotating the gripper to a target quaternion (e.g., 90 degrees)
                # Use a default target_quat if not available
                target_quat = np.array([0.0, 0.0, 0.7071, 0.7071])  # 90 deg about z
                print(f"[Exploration] Trying rotate to {target_quat}")
                obs, reward, done = rotate(env, task, target_quat)
                exploration_results['rotate'] = 'success'
            except Exception as e:
                print(f"[Exploration] rotate failed: {e}")
                exploration_results['rotate'] = f'failed: {e}'

        # 5. Try 'pull' skill
        if 'pull' in available_skills:
            try:
                # Try pulling a drawer or object if available
                drawer_keys = [k for k in positions.keys() if 'drawer' in k]
                if drawer_keys:
                    drawer = drawer_keys[0]
                    print(f"[Exploration] Trying pull on {drawer}")
                    obs, reward, done = pull(env, task, drawer)
                    exploration_results['pull'] = 'success'
                else:
                    print("[Exploration] No drawer found to test pull.")
                    exploration_results['pull'] = 'skipped'
            except Exception as e:
                print(f"[Exploration] pull failed: {e}")
                exploration_results['pull'] = f'failed: {e}'

        print("=== Exploration Results ===")
        for skill, result in exploration_results.items():
            print(f"Skill '{skill}': {result}")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this code, the focus is on exploration to find missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()