# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'bottom_drawer': ..., 'gripper': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate (is-open bottom) ---
        # Feedback indicates that the predicate (is-open bottom) is missing.
        # We need to explore the state of the 'bottom' drawer to determine if it is open.
        # The exploration domain suggests that 'pull' can be used to learn about lock/open state.

        # For this example, we assume the following object keys exist:
        # 'bottom_drawer' : the drawer object
        # 'gripper' : the robot's gripper
        # 'bottom_drawer_handle' : the position to grasp/pull the drawer
        # 'bottom_drawer_side' : a side position for the drawer
        # 'bottom_drawer_anchor' : anchor position for the drawer
        # 'ninety_deg' : angle for rotation (if needed)
        # These keys may need to be adjusted based on your environment.

        # --- Step 1: Move gripper to side position of bottom drawer ---
        try:
            bottom_drawer_side_pos = positions['bottom_drawer_side']
            print("[Exploration] Moving gripper to side position of bottom drawer:", bottom_drawer_side_pos)
            obs, reward, done = move(env, task, target_pos=bottom_drawer_side_pos)
            if done:
                print("[Exploration] Task ended during move to side position!")
                return
        except Exception as e:
            print("[Error] Could not move to side position:", e)
            return

        # --- Step 2: Rotate gripper to 90 degrees (if required) ---
        try:
            # If your rotate skill requires gripper name/id and angles, provide them accordingly.
            # For example: rotate(env, task, gripper_name, from_angle, to_angle)
            gripper_name = 'gripper'
            from_angle = 'zero_deg'
            to_angle = 'ninety_deg'
            print("[Exploration] Rotating gripper from", from_angle, "to", to_angle)
            obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
            if done:
                print("[Exploration] Task ended during gripper rotation!")
                return
        except Exception as e:
            print("[Error] Could not rotate gripper:", e)
            return

        # --- Step 3: Move gripper to anchor position of bottom drawer ---
        try:
            bottom_drawer_anchor_pos = positions['bottom_drawer_anchor']
            print("[Exploration] Moving gripper to anchor position of bottom drawer:", bottom_drawer_anchor_pos)
            obs, reward, done = move(env, task, target_pos=bottom_drawer_anchor_pos)
            if done:
                print("[Exploration] Task ended during move to anchor position!")
                return
        except Exception as e:
            print("[Error] Could not move to anchor position:", e)
            return

        # --- Step 4: Pick the bottom drawer (grasp handle) ---
        try:
            # If your pick skill requires object name and position, provide them accordingly.
            # For example: pick(env, task, target_pos, approach_distance, ...)
            bottom_drawer_handle_pos = positions['bottom_drawer_handle']
            print("[Exploration] Picking (grasping) bottom drawer at handle position:", bottom_drawer_handle_pos)
            obs, reward, done = pick(env, task, target_pos=bottom_drawer_handle_pos, approach_distance=0.10)
            if done:
                print("[Exploration] Task ended during pick (grasp)!")
                return
        except Exception as e:
            print("[Error] Could not pick (grasp) bottom drawer:", e)
            return

        # --- Step 5: Pull the bottom drawer to check if it is open ---
        try:
            # The pull skill may require the gripper and drawer names/ids.
            gripper_name = 'gripper'
            drawer_name = 'bottom_drawer'
            print("[Exploration] Pulling the bottom drawer to check open state.")
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print("[Error] Could not pull bottom drawer:", e)
            return

        # --- Step 6: Check if the bottom drawer is open (predicate is-open bottom) ---
        # This is the exploration result: after pulling, we can check the state.
        try:
            # The observation or environment may provide a way to check if the drawer is open.
            # For example, obs might have a field like obs.is_open_bottom or similar.
            is_open = False
            if hasattr(obs, 'is_open_bottom'):
                is_open = obs.is_open_bottom
            elif hasattr(obs, 'drawer_states'):
                # drawer_states could be a dict or list; adjust as needed
                is_open = obs.drawer_states.get('bottom', False)
            else:
                # Fallback: try to get from environment or positions
                is_open = positions.get('bottom_drawer_is_open', False)
            print(f"[Exploration] Is the bottom drawer open? {is_open}")
        except Exception as e:
            print("[Error] Could not determine if bottom drawer is open:", e)

        # --- End of Exploration Phase ---

        # At this point, the missing predicate (is-open bottom) has been explored.
        # You can now proceed with the rest of your oracle plan as needed,
        # using only the available skills: pick, place, move, rotate, pull.

        # (If there are further steps in the oracle plan, add them here.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
