# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to Discover Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to execute each available skill and observe the environment for changes
        # that may indicate the presence of a missing predicate (e.g., lock-known, identified, etc.)

        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']
        exploration_results = {}

        # For demonstration, we will attempt to pick and pull on all objects, and observe any errors or state changes.
        # This is a generic exploration; in a real scenario, you would tailor this to your domain and objects.

        print("=== [Exploration] Starting exploration of available skills to detect missing predicates ===")
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting actions on object: {obj_name} at {obj_pos}")
            # Try move (move gripper to object)
            try:
                print(f"  [Exploration] move to {obj_name}")
                obs, reward, done = move(env, task, np.array(obj_pos))
                if done:
                    print(f"  [Exploration] Task ended during move to {obj_name}")
                    break
            except Exception as e:
                print(f"  [Exploration] move failed for {obj_name}: {e}")

            # Try pick
            try:
                print(f"  [Exploration] pick {obj_name}")
                obs, reward, done = pick(env, task, target_pos=np.array(obj_pos))
                if done:
                    print(f"  [Exploration] Task ended during pick of {obj_name}")
                    break
            except Exception as e:
                print(f"  [Exploration] pick failed for {obj_name}: {e}")

            # Try pull (if available)
            try:
                print(f"  [Exploration] pull {obj_name}")
                obs, reward, done = pull(env, task, target_pos=np.array(obj_pos))
                if done:
                    print(f"  [Exploration] Task ended during pull of {obj_name}")
                    break
            except Exception as e:
                print(f"  [Exploration] pull failed for {obj_name}: {e}")

            # Try place (if holding)
            try:
                print(f"  [Exploration] place {obj_name}")
                obs, reward, done = place(env, task, target_pos=np.array(obj_pos))
                if done:
                    print(f"  [Exploration] Task ended during place of {obj_name}")
                    break
            except Exception as e:
                print(f"  [Exploration] place failed for {obj_name}: {e}")

            # Try rotate (if available)
            try:
                print(f"  [Exploration] rotate {obj_name}")
                # For rotate, we need to know the gripper and angles; here we use dummy values for exploration
                obs, reward, done = rotate(env, task, gripper_name='gripper', from_angle=0, to_angle=90)
                if done:
                    print(f"  [Exploration] Task ended during rotate of {obj_name}")
                    break
            except Exception as e:
                print(f"  [Exploration] rotate failed for {obj_name}: {e}")

        print("=== [Exploration] Exploration phase complete. Check logs for any clues about missing predicates. ===")

        # === End of Exploration Phase ===
        # At this point, you would analyze the logs/output to determine which predicate is missing.
        # For this code, we simply perform the exploration and log the results.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()