# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to use available skills to interact with objects and observe state changes,
        # to infer which predicate is missing (e.g., lock-known, identified, etc.)

        # For demonstration, we will:
        # 1. Move to each object location (if possible)
        # 2. Attempt to pick/place/pull/rotate as allowed by the skills
        # 3. Log observations and errors to help identify missing predicates

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # Try to explore all objects in the environment
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting to interact with object: {obj_name} at {obj_pos}")

            # 1. Move to the object's position
            try:
                print(f"[Exploration] Moving to {obj_name} position...")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended during move to {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during move to {obj_name}: {e}")

            # 2. Try to pick the object (if possible)
            try:
                print(f"[Exploration] Attempting to pick {obj_name}...")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pick of {obj_name}: {e}")

            # 3. Try to place the object back (if possible)
            try:
                print(f"[Exploration] Attempting to place {obj_name}...")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during place of {obj_name}: {e}")

            # 4. Try to rotate (if the object is a gripper or has a rotatable part)
            try:
                print(f"[Exploration] Attempting to rotate at {obj_name}...")
                # For demonstration, try rotating from 'zero_deg' to 'ninety_deg' if such angles exist
                # These are placeholders; in a real environment, you would query available angles
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name='gripper',  # Replace with actual gripper name if available
                    from_angle='zero_deg',
                    to_angle='ninety_deg'
                )
                if done:
                    print(f"[Exploration] Task ended after rotate at {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during rotate at {obj_name}: {e}")

            # 5. Try to pull (if the object is a drawer or similar)
            try:
                print(f"[Exploration] Attempting to pull {obj_name}...")
                obs, reward, done = pull(
                    env,
                    task,
                    drawer_name=obj_name  # Assuming the object is a drawer
                )
                if done:
                    print(f"[Exploration] Task ended after pull of {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pull of {obj_name}: {e}")

        print("[Exploration] Completed exploration of all objects.")

        # === End of Exploration Phase ===
        # After running this, check the logs for exceptions or failed actions.
        # If a particular action always fails (e.g., pull always fails due to missing 'lock-known' predicate),
        # this suggests which predicate is missing in the domain/problem description.

        # TODO: After identifying the missing predicate, update the domain/problem PDDL accordingly
        # and re-run the planner.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()