# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback suggests a missing predicate is causing planning to fail.
        # We will attempt to explore the environment using available skills to trigger state changes
        # and observe if any new information (predicates) can be inferred.

        # For demonstration, we will:
        # 1. Move to each object position (if possible)
        # 2. Attempt to pick and place objects (if possible)
        # 3. Try rotate and pull actions (if possible)
        # 4. Log any errors or unexpected results

        # The available skills are: ['pick', 'place', 'move', 'rotate', 'pull']

        # Get all object names and their positions
        object_names = list(positions.keys())
        print("[Exploration] Found objects:", object_names)

        # Try to move to each object's position
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to {obj_name} at {obj_pos}")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(obj_pos))
                if done:
                    print(f"[Exploration] Task ended after moving to {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Error during move to {obj_name}: {e}")

        # Try to pick each object (if possible)
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to pick {obj_name} at {obj_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Error during pick of {obj_name}: {e}")

        # Try to place each object at its original position (if possible)
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to place {obj_name} at {obj_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Error during place of {obj_name}: {e}")

        # Try to rotate and pull (if possible)
        # For rotate, we need gripper, from_angle, to_angle; for pull, we need object
        # We'll attempt generic calls with placeholder values if possible
        # These may fail if the environment does not support them for the current objects

        # Attempt rotate on possible grippers/angles
        possible_angles = ['zero_deg', 'ninety_deg']
        for gripper_name in object_names:
            for from_angle in possible_angles:
                for to_angle in possible_angles:
                    if from_angle == to_angle:
                        continue
                    print(f"[Exploration] Attempting to rotate {gripper_name} from {from_angle} to {to_angle}")
                    try:
                        obs, reward, done = rotate(
                            env,
                            task,
                            gripper_name,
                            from_angle,
                            to_angle
                        )
                        if done:
                            print(f"[Exploration] Task ended after rotate {gripper_name}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Error during rotate of {gripper_name}: {e}")

        # Attempt pull on each object
        for obj_name in object_names:
            print(f"[Exploration] Attempting to pull {obj_name}")
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    obj_name
                )
                if done:
                    print(f"[Exploration] Task ended after pull {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Error during pull of {obj_name}: {e}")

        print("[Exploration] Completed all available skill explorations.")

        # === End of Exploration Phase ===

        # At this point, you should review the logs and environment state to infer which predicate is missing.
        # This exploration phase is designed to help you identify what knowledge or state is not being captured,
        # as suggested by the feedback and exploration knowledge.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()