# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'bottom_drawer': (x, y, z), ...}
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (is-open bottom)
        # The goal is to determine if the 'bottom' drawer is open.
        # We need to explore/interact with the 'bottom' drawer to discover its state.

        # Assumptions for object naming:
        # - The drawer is named 'bottom' or 'bottom_drawer' in positions.
        # - The gripper is named 'gripper' or similar.
        # - The environment provides the necessary positions.

        # Try to find the bottom drawer position
        drawer_key = None
        for k in positions:
            if 'bottom' in k and 'drawer' in k:
                drawer_key = k
                break
        if drawer_key is None:
            # Fallback: try just 'bottom'
            for k in positions:
                if k == 'bottom':
                    drawer_key = k
                    break
        if drawer_key is None:
            print("[Exploration] Could not find bottom drawer in object positions.")
            return

        bottom_drawer_pos = positions[drawer_key]
        print(f"[Exploration] Bottom drawer position: {bottom_drawer_pos}")

        # Move the gripper to the bottom drawer's position to explore
        try:
            print("[Exploration] Moving to bottom drawer to explore its state...")
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bottom_drawer_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to bottom drawer!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            return

        # Try to rotate the gripper if needed (assume 'rotate' skill rotates to required angle)
        # For exploration, we may want to rotate to 'ninety_deg' if required by the domain
        # We'll attempt to rotate if the skill is available and the environment supports it
        try:
            print("[Exploration] Rotating gripper to ninety_deg (if required)...")
            # The rotate skill may require gripper name and angle names
            # We'll use default names if possible
            gripper_name = 'gripper'
            from_angle = 'zero_deg'
            to_angle = 'ninety_deg'
            obs, reward, done = rotate(
                env,
                task,
                gripper_name,
                from_angle,
                to_angle
            )
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate (may be non-critical): {e}")

        # Try to pull the drawer to check if it is openable (and thus discover is-open)
        try:
            print("[Exploration] Attempting to pull the bottom drawer to check if it is open...")
            # The pull skill may require gripper and drawer names
            obs, reward, done = pull(
                env,
                task,
                gripper_name,
                'bottom'  # or drawer_key if that's the correct name
            )
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull (may be non-critical): {e}")

        # After exploration, print out the result (in a real system, would check predicates)
        print("[Exploration] Exploration phase complete. If the drawer opened, (is-open bottom) is now true.")

        # --- End of Exploration Phase ---

        # (Optional) Continue with further task logic if needed

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()