# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase for Missing Predicate ===
        # The feedback indicates a missing predicate is likely required for planning.
        # We will perform a sequence of available skills to explore the environment and infer which predicate is missing.
        # The available skills are: ['pick', 'place', 'move', 'rotate', 'pull']

        # 1. List all objects and their positions for exploration
        print("[Exploration] Available objects and positions:")
        for obj_name, pos in positions.items():
            print(f"    {obj_name}: {pos}")

        # 2. Try to move to each object's position and attempt pick/place/pull/rotate as appropriate
        #    This will help us determine if any predicate (e.g., 'identified', 'weight-known', 'durability-known', 'lock-known') is missing.
        #    We will log the results of each action.

        # For demonstration, we will attempt to:
        #   - Move to each object
        #   - Pick the object (if possible)
        #   - Place the object at a new location (if possible)
        #   - Try pull/rotate if the object is a drawer or similar

        # For simplicity, we assume the robot's gripper starts at a default position.
        # We'll use the first object as a test case for pick/place, and if a drawer is present, for pull/rotate.

        # Find a non-drawer object for pick/place, and a drawer for pull/rotate
        non_drawer_obj = None
        non_drawer_pos = None
        drawer_obj = None
        drawer_pos = None

        for obj_name, pos in positions.items():
            if 'drawer' in obj_name.lower():
                drawer_obj = obj_name
                drawer_pos = pos
            else:
                non_drawer_obj = obj_name
                non_drawer_pos = pos

        # 3. Exploration: Try to pick and place a non-drawer object
        if non_drawer_obj is not None:
            print(f"[Exploration] Attempting to move to and pick object: {non_drawer_obj}")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(non_drawer_pos))
                if done:
                    print("[Exploration] Task ended during move to object!")
                    return

                obs, reward, done = pick(env, task, target_pos=np.array(non_drawer_pos))
                if done:
                    print("[Exploration] Task ended during pick!")
                    return

                # Try to place at a slightly offset position
                place_pos = np.array(non_drawer_pos) + np.array([0.1, 0, 0])
                print(f"[Exploration] Attempting to place object: {non_drawer_obj} at {place_pos}")
                obs, reward, done = place(env, task, target_pos=place_pos)
                if done:
                    print("[Exploration] Task ended during place!")
                    return

            except Exception as e:
                print(f"[Exploration] Exception during pick/place exploration: {e}")

        # 4. Exploration: Try to pull/rotate a drawer if present
        if drawer_obj is not None:
            print(f"[Exploration] Attempting to move to and pull/rotate drawer: {drawer_obj}")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(drawer_pos))
                if done:
                    print("[Exploration] Task ended during move to drawer!")
                    return

                # Try to rotate (if skill is available and makes sense)
                try:
                    # For rotate, we need gripper, from_angle, to_angle; we use dummy values for exploration
                    obs, reward, done = rotate(env, task, gripper_name='gripper', from_angle='zero_deg', to_angle='ninety_deg')
                    if done:
                        print("[Exploration] Task ended during rotate!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during rotate: {e}")

                # Try to pull
                try:
                    obs, reward, done = pull(env, task, drawer_name=drawer_obj)
                    if done:
                        print("[Exploration] Task ended during pull!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during pull: {e}")

            except Exception as e:
                print(f"[Exploration] Exception during drawer exploration: {e}")

        # 5. Exploration summary
        print("[Exploration] Exploration phase complete. If any actions failed due to missing predicates, check logs above for clues.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()