# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer1': ..., 'gripper': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Check if drawer1 is locked ---
        # Feedback indicates we need to discover (not (is-locked drawer1))
        # We use the available skills to try to pull the drawer and see if it is locked.
        # If pull succeeds, then drawer is not locked.

        drawer_name = 'drawer1'
        gripper_name = 'gripper'
        # The following assumes positions contains keys for drawer1 and gripper
        try:
            drawer_pos = positions[drawer_name]
            gripper_pos = positions[gripper_name]
        except KeyError as e:
            print(f"[Error] Missing object in positions: {e}")
            shutdown_environment(env)
            return

        # 1. Move gripper to the side position of the drawer (simulate approach)
        print(f"[Exploration] Moving gripper to side of {drawer_name} at {drawer_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=drawer_pos)
            if done:
                print("[Exploration] Task ended during move to drawer side!")
                return
        except Exception as e:
            print(f"[Error] Exception during move: {e}")
            shutdown_environment(env)
            return

        # 2. Rotate gripper to required angle (assume 'ninety_deg' is required for side approach)
        # We need to know the current angle and the target angle. For this example, we use names.
        # In a real environment, you would get the current angle from obs.
        from_angle = 'zero_deg'
        to_angle = 'ninety_deg'
        print(f"[Exploration] Rotating gripper from {from_angle} to {to_angle}")
        try:
            obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # 3. Attempt to pull the drawer to check if it is locked
        print(f"[Exploration] Attempting to pull {drawer_name} to check lock status")
        try:
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Exploration] Task ended during pull!")
                return
            # If pull succeeds, we know (not (is-locked drawer1))
            print(f"[Exploration] Pull succeeded: {drawer_name} is NOT locked.")
        except Exception as e:
            print(f"[Exploration] Pull failed: {drawer_name} may be locked or another error occurred. Exception: {e}")
            # If pull fails, we assume drawer is locked (not the case in this feedback)
            shutdown_environment(env)
            return

        # --- Main Task Plan (after exploration) ---
        # Now that we know drawer1 is not locked, proceed with the oracle plan.
        # For demonstration, let's assume the plan is to open the drawer and place an object inside.

        # 4. (Optional) Pick up an object to place in the drawer
        object_to_pick = None
        for obj in positions:
            if obj not in [drawer_name, gripper_name]:
                object_to_pick = obj
                break
        if object_to_pick is not None:
            object_pos = positions[object_to_pick]
            print(f"[Task] Picking up {object_to_pick} at {object_pos}")
            try:
                obs, reward, done = move(env, task, target_pos=object_pos)
                if done:
                    print("[Task] Task ended during move to object!")
                    return
                obs, reward, done = pick(env, task, object_to_pick, 'table')  # Assume 'table' as location
                if done:
                    print("[Task] Task ended during pick!")
                    return
            except Exception as e:
                print(f"[Error] Exception during pick: {e}")
                shutdown_environment(env)
                return

        # 5. Move to drawer and place the object inside
        print(f"[Task] Moving to {drawer_name} to place object inside")
        try:
            obs, reward, done = move(env, task, target_pos=drawer_pos)
            if done:
                print("[Task] Task ended during move to drawer!")
                return
            if object_to_pick is not None:
                obs, reward, done = place(env, task, object_to_pick, drawer_name)
                if done:
                    print("[Task] Task ended during place!")
                    return
        except Exception as e:
            print(f"[Error] Exception during place: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()