# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'tomato2': (x, y, z), 'table1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at tomato2 table1) is present in the initial state, but the plan fails.
        # Exploration knowledge suggests that the robot may need to "identify" the object before acting.
        # We simulate this by first "moving" to the location to trigger identification.

        # For this example, we assume the robot must "move" to table1 to identify tomato2 before picking.
        # The available skills include 'move', but the actual implementation may be a no-op or handled in the environment.
        # We use the skill as per the available list.

        # --- Define object and location names based on feedback and positions ---
        tomato_name = 'tomato2'
        table_name = 'table1'

        # Defensive: Check if required objects are in the positions dictionary
        if tomato_name not in positions or table_name not in positions:
            print(f"[Error] Required object or location not found in positions: {tomato_name}, {table_name}")
            return

        tomato_pos = positions[tomato_name]
        table_pos = positions[table_name]

        # --- Step 1: Move to table1 to "identify" tomato2 (exploration phase) ---
        print(f"[Exploration] Moving to {table_name} to identify {tomato_name}...")
        try:
            # The move skill may require from_pos and to_pos; we use current gripper position as from_pos
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            # Call the move skill (if implemented as such)
            obs, reward, done = move(env, task, from_pos=gripper_pos, to_pos=table_pos)
            if done:
                print("[Exploration] Task ended during move!")
                return
        except Exception as e:
            print(f"[Warning] Move skill failed or not implemented: {e}")
            # If move is not implemented, continue to pick

        # --- Step 2: Pick up tomato2 at table1 ---
        print(f"[Task] Picking up {tomato_name} at {table_name} (pos: {tomato_pos})...")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Error] Pick skill failed: {e}")
            return

        # --- Step 3: (Optional) Place tomato2 somewhere else if required ---
        # If the plan requires placing the object, add a place step here.
        # For demonstration, we check if a 'place' location is available.
        place_location_name = None
        for key in positions:
            if key != tomato_name and key != table_name:
                place_location_name = key
                break
        if place_location_name:
            place_pos = positions[place_location_name]
            print(f"[Task] Placing {tomato_name} at {place_location_name} (pos: {place_pos})...")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Error] Place skill failed: {e}")
                return

        print("[Task] Skeleton task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()