# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (rotated gripper1 zero_deg)
        # The goal is to check if the predicate (rotated gripper1 zero_deg) is true in the initial state.
        # If not, we need to rotate the gripper to zero_deg using the rotate skill.

        # For this, we need to know:
        # - The gripper name (assume 'gripper1' as per feedback)
        # - The current orientation of the gripper
        # - The target orientation corresponding to 'zero_deg'

        # Step 1: Get current gripper orientation
        try:
            obs = task.get_observation()
            current_quat = obs.gripper_pose[3:7]
        except Exception as e:
            print("[Exploration] Error retrieving gripper pose:", e)
            current_quat = None

        # Step 2: Define target quaternion for 'zero_deg'
        # This mapping may depend on your environment; here we assume zero_deg is [0,0,0,1]
        target_quat_zero_deg = np.array([0.0, 0.0, 0.0, 1.0])

        # Step 3: Check if current orientation matches zero_deg (within threshold)
        def quat_angle_diff(q1, q2):
            q1 = np.array(q1) / np.linalg.norm(q1)
            q2 = np.array(q2) / np.linalg.norm(q2)
            dot = np.dot(q1, q2)
            if dot < 0:
                dot = -dot
            angle = 2 * np.arccos(np.clip(dot, -1.0, 1.0))
            return angle

        need_rotate = True
        if current_quat is not None:
            angle_diff = quat_angle_diff(current_quat, target_quat_zero_deg)
            print(f"[Exploration] Gripper angle diff to zero_deg: {angle_diff:.3f} rad")
            if angle_diff < 0.05:
                print("[Exploration] Gripper already at zero_deg.")
                need_rotate = False

        # Step 4: If not at zero_deg, use rotate skill to set it
        if need_rotate:
            print("[Exploration] Rotating gripper to zero_deg...")
            try:
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat_zero_deg,
                    max_steps=100,
                    threshold=0.05,
                    timeout=10.0
                )
                print("[Exploration] Gripper rotated to zero_deg.")
            except Exception as e:
                print("[Exploration] Error during rotate skill:", e)
        else:
            print("[Exploration] No rotation needed.")

        # === Main Task Plan ===
        # At this point, the missing predicate (rotated gripper1 zero_deg) is ensured.
        # Insert further plan steps here as needed, using only the available skills:
        # ['pick', 'place', 'move', 'rotate', 'pull']

        # Example: (Commented out, as actual plan is not specified)
        # obj_name = 'object_1'
        # obj_pos = positions.get(obj_name)
        # if obj_pos is not None:
        #     print(f"[Task] Picking {obj_name} at {obj_pos}")
        #     try:
        #         obs, reward, done = pick(env, task, target_pos=obj_pos)
        #         if done:
        #             print("[Task] Task ended after pick!")
        #             return
        #     except Exception as e:
        #         print(f"[Task] Error during pick: {e}")

        # TODO: Add further plan steps as required by your specific task.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
