# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # Based on feedback, we need to explore the environment to find out which predicate is missing.
        # We do this by attempting to use the available skills on all objects and locations.

        # Get all object names and their positions
        object_names = list(positions.keys())
        print("[Exploration] Objects found in environment:", object_names)

        # Try to move to each object's position and pick/place/rotate/pull as appropriate
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to {obj_name} at {obj_pos}")
            try:
                # Move to the object's position
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended during move to {obj_name}!")
                    return

                # Try to pick the object
                print(f"[Exploration] Attempting to pick {obj_name}")
                try:
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during pick of {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Pick failed for {obj_name}: {e}")

                # Try to place the object back (if pick succeeded)
                print(f"[Exploration] Attempting to place {obj_name}")
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during place of {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Place failed for {obj_name}: {e}")

                # Try to rotate (if applicable)
                print(f"[Exploration] Attempting to rotate at {obj_name}")
                try:
                    # For rotate, we need to know gripper, from_angle, to_angle
                    # Here, we use dummy values as placeholders; in a real scenario, these should be determined from the environment
                    gripper = 'gripper'
                    from_angle = 'zero_deg'
                    to_angle = 'ninety_deg'
                    obs, reward, done = rotate(
                        env,
                        task,
                        gripper=gripper,
                        from_angle=from_angle,
                        to_angle=to_angle
                    )
                    if done:
                        print(f"[Exploration] Task ended during rotate at {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Rotate failed for {obj_name}: {e}")

                # Try to pull (if applicable)
                print(f"[Exploration] Attempting to pull {obj_name}")
                try:
                    # For pull, we need gripper and drawer/object
                    gripper = 'gripper'
                    drawer = obj_name  # Assume object is a drawer for exploration
                    obs, reward, done = pull(
                        env,
                        task,
                        gripper=gripper,
                        drawer=drawer
                    )
                    if done:
                        print(f"[Exploration] Task ended during pull of {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Pull failed for {obj_name}: {e}")

            except Exception as e:
                print(f"[Exploration] Move failed for {obj_name}: {e}")

        print("[Exploration] Completed exploration phase. If the planner still fails, check logs for missing predicates or actions.")

        # === (Optional) Main Task Plan ===
        # After exploration, you can implement the actual oracle plan here using the available skills.
        # For now, the focus is on exploration to identify missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()