# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (rotated gripper1 zero_deg)
        # The missing predicate is likely related to the initial orientation of the gripper.
        # We will perform an exploration step to check and, if needed, rotate the gripper to zero_deg.

        # Assumptions for this code:
        # - There is a gripper named 'gripper1'
        # - The angle 'zero_deg' is a known orientation (e.g., quaternion or euler)
        # - The rotate skill expects (env, task, target_quat, ...)

        # 1. Get the current orientation of gripper1
        try:
            obs = task.get_observation()
            current_quat = obs.gripper_pose[3:7]
            # Define the target quaternion for zero_deg
            # This may need to be adapted to your environment's convention
            # For example, zero_deg = [0, 0, 0, 1] (identity quaternion)
            zero_deg_quat = np.array([0, 0, 0, 1], dtype=np.float32)
            # Check if current orientation is already zero_deg (within threshold)
            dot_product = np.dot(current_quat, zero_deg_quat)
            if dot_product < 0:
                dot_product = -dot_product
            angle_diff = 2 * np.arccos(np.clip(dot_product, -1.0, 1.0))
            threshold = 0.05  # radians
            if angle_diff > threshold:
                print("[Exploration] Gripper is not at zero_deg, rotating to zero_deg.")
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=zero_deg_quat,
                    max_steps=100,
                    threshold=threshold,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during rotate!")
                    return
            else:
                print("[Exploration] Gripper already at zero_deg orientation.")
        except Exception as e:
            print(f"[Exploration] Exception during gripper orientation check/rotate: {e}")

        # === Main Task Plan ===
        # Insert your oracle plan here, using only the available skills:
        # ['pick', 'place', 'move', 'rotate', 'pull']
        #
        # Example (replace with your actual plan as needed):

        # Example: Pick up an object and place it somewhere else
        # (Replace 'object_1', 'location_A', 'location_B' with actual names from your environment)
        try:
            object_1 = 'object_1'
            location_A = 'location_A'
            location_B = 'location_B'
            # If these names are not in positions, skip this example
            if object_1 in positions and location_A in positions and location_B in positions:
                print(f"[Task] Picking {object_1} at {location_A}")
                obs, reward, done = pick(
                    env,
                    task,
                    obj=object_1,
                    loc=location_A
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return

                print(f"[Task] Placing {object_1} at {location_B}")
                obs, reward, done = place(
                    env,
                    task,
                    obj=object_1,
                    loc=location_B
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            else:
                print("[Task] Example objects/locations not found in positions, skipping main plan.")
        except Exception as e:
            print(f"[Task] Exception during main plan execution: {e}")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
