# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may need to "identify" an object
        # or learn about its properties (e.g., weight, durability, lock-known) before manipulation.
        # We'll attempt to "move" to the location of each object to trigger identification predicates,
        # and then try to "pick" or "pull" as appropriate.

        # For demonstration, let's assume the environment contains a drawer and an object (e.g., tomato1)
        # and the robot must open the drawer and pick the object.

        # We'll use only the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # Example object and drawer names (replace with actual names from your environment if different)
        object_name = 'tomato1'
        drawer_name = 'drawer1'
        gripper_name = 'gripper1'
        plate_name = 'plate'
        # Positions (replace with actual keys if different)
        try:
            object_pos = positions[object_name]
        except KeyError:
            print(f"[Error] Object '{object_name}' not found in positions. Available: {list(positions.keys())}")
            object_pos = None
        try:
            drawer_pos = positions[drawer_name]
        except KeyError:
            print(f"[Error] Drawer '{drawer_name}' not found in positions. Available: {list(positions.keys())}")
            drawer_pos = None
        try:
            plate_pos = positions[plate_name]
        except KeyError:
            print(f"[Error] Plate '{plate_name}' not found in positions. Available: {list(positions.keys())}")
            plate_pos = None

        # === Exploration: Try to identify the object by moving to its location ===
        # (In a real scenario, you may need to iterate over all objects/locations)
        # Here, we simulate the exploration by moving to the object's location.
        # If the move action is not available as a skill, skip this step.

        if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
            try:
                print(f"[Exploration] Moving to {object_name} at {object_pos} to identify it.")
                # The move skill may require from/to locations; here we use positions as a proxy.
                # If your move skill requires location names, adjust accordingly.
                obs, reward, done = move(env, task, from_pos=None, to_pos=object_pos)
                if done:
                    print("[Exploration] Task ended during move!")
                    return
            except Exception as e:
                print(f"[Exploration] Move to {object_name} failed: {e}")

        # === Attempt to pick the object (may reveal missing predicates) ===
        if 'pick' in ['pick', 'place', 'move', 'rotate', 'pull']:
            try:
                print(f"[Exploration] Attempting to pick {object_name} at {object_pos}.")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=object_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Exploration] Pick of {object_name} failed: {e}")

        # === If the object is inside a drawer, try to open the drawer first ===
        # This may require rotating, moving the gripper, picking the drawer handle, and pulling.
        # We'll attempt these steps using the available skills.

        # 1. Rotate gripper to required angle (e.g., 90 degrees)
        if 'rotate' in ['pick', 'place', 'move', 'rotate', 'pull']:
            try:
                print(f"[Exploration] Rotating gripper for drawer manipulation.")
                # Example: rotate to 90 degrees (quaternion for 90 deg about z)
                from scipy.spatial.transform import Rotation as R
                target_euler = [0, 0, np.pi/2]  # 90 deg about z
                target_quat = R.from_euler('xyz', target_euler).as_quat()
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=100,
                    threshold=0.05,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after rotate!")
                    return
            except Exception as e:
                print(f"[Exploration] Rotate failed: {e}")

        # 2. Pull the drawer (if possible)
        if 'pull' in ['pick', 'place', 'move', 'rotate', 'pull']:
            try:
                print(f"[Exploration] Attempting to pull {drawer_name}.")
                obs, reward, done = pull(
                    env,
                    task,
                    drawer_name
                )
                if done:
                    print("[Exploration] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Exploration] Pull of {drawer_name} failed: {e}")

        # 3. After opening, try to pick the object again (if not already picked)
        if 'pick' in ['pick', 'place', 'move', 'rotate', 'pull']:
            try:
                print(f"[Exploration] Attempting to pick {object_name} after opening drawer.")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=object_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after pick (post-drawer)!")
                    return
            except Exception as e:
                print(f"[Exploration] Pick of {object_name} after drawer open failed: {e}")

        # 4. Place the object on the plate (if available)
        if 'place' in ['pick', 'place', 'move', 'rotate', 'pull'] and plate_pos is not None:
            try:
                print(f"[Exploration] Attempting to place {object_name} on {plate_name}.")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=plate_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Exploration] Place of {object_name} on {plate_name} failed: {e}")

        print("[Exploration] Completed exploration phase. If the task did not succeed, check for missing predicates or required state knowledge.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()