# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (is-locked) ===
        # Feedback: (not (is-locked drawer1))
        # We need to determine if drawer1 is locked or not before attempting to pull/open it.
        # The only available skills are: ['pick', 'place', 'move', 'rotate', 'pull']
        # According to the exploration knowledge, the 'pull' action can be used to discover the lock state.

        # --- Identify relevant objects and positions ---
        # These keys should match your environment's naming conventions.
        # For this example, we assume:
        #   - The drawer is named 'drawer1'
        #   - The gripper is 'gripper1'
        #   - The handle position is 'drawer1_handle'
        #   - The anchor position for the gripper is 'drawer1_anchor'
        #   - The side position for the gripper is 'drawer1_side'
        #   - The angle names are 'zero_deg' and 'ninety_deg'
        #   - The pull direction is along the 'x' axis (adjust as needed)
        #   - The object to pick is 'drawer1_handle' (if needed)
        #   - The robot starts at a default position

        # You may need to adjust these keys to match your environment's object naming.
        drawer_name = 'drawer1'
        gripper_name = 'gripper1'
        handle_name = 'drawer1_handle'
        anchor_pos_name = 'drawer1_anchor'
        side_pos_name = 'drawer1_side'
        zero_angle = 'zero_deg'
        ninety_angle = 'ninety_deg'
        pull_axis = 'x'
        pull_distance = 0.15  # Example value; adjust as needed

        # --- Step 1: Move gripper to side position and rotate to 90 degrees ---
        try:
            print(f"[Exploration] Moving gripper '{gripper_name}' to side position '{side_pos_name}' and rotating to {ninety_angle}.")
            # Move gripper to side position (if needed, use 'move' or 'move-to-side' if available)
            # Here, we use the 'move' skill if it is defined for gripper movement.
            if 'move' in globals():
                # If move expects (env, task, from_pos, to_pos), get current and target positions
                current_pos = positions.get(f"{gripper_name}_pos", None)
                side_pos = positions.get(side_pos_name, None)
                if current_pos is not None and side_pos is not None:
                    obs, reward, done = move(env, task, current_pos, side_pos)
                    if done:
                        print("[Exploration] Task ended during move to side position!")
                        return
            # Rotate gripper to 90 degrees
            if 'rotate' in globals():
                obs, reward, done = rotate(env, task, gripper_name, zero_angle, ninety_angle)
                if done:
                    print("[Exploration] Task ended during rotate!")
                    return
        except Exception as e:
            print(f"[Exploration] Exception during move/rotate: {e}")

        # --- Step 2: Move gripper from side to anchor position ---
        try:
            print(f"[Exploration] Moving gripper '{gripper_name}' from side '{side_pos_name}' to anchor '{anchor_pos_name}'.")
            # If move-to-anchor is not a skill, use 'move' if it moves gripper
            if 'move' in globals():
                side_pos = positions.get(side_pos_name, None)
                anchor_pos = positions.get(anchor_pos_name, None)
                if side_pos is not None and anchor_pos is not None:
                    obs, reward, done = move(env, task, side_pos, anchor_pos)
                    if done:
                        print("[Exploration] Task ended during move to anchor position!")
                        return
        except Exception as e:
            print(f"[Exploration] Exception during move to anchor: {e}")

        # --- Step 3: Pick the drawer handle (if required) ---
        try:
            print(f"[Exploration] Picking drawer handle '{handle_name}' at anchor position '{anchor_pos_name}'.")
            if 'pick' in globals():
                anchor_pos = positions.get(anchor_pos_name, None)
                if anchor_pos is not None:
                    obs, reward, done = pick(env, task, handle_name, anchor_pos)
                    if done:
                        print("[Exploration] Task ended during pick!")
                        return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")

        # --- Step 4: Exploration - Pull to discover lock state ---
        try:
            print(f"[Exploration] Pulling drawer '{drawer_name}' to check lock state.")
            # Use the pull skill to attempt to pull the drawer and discover if it is locked.
            # The pull skill signature is: pull(env, task, pull_distance, pull_axis, ...)
            obs, reward, done = pull(env, task, pull_distance, pull_axis)
            if done:
                print("[Exploration] Task ended during pull!")
                return
            print("[Exploration] Pull action completed. Lock state should now be known.")
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")

        # --- Step 5: (Optional) Continue with main task plan if lock state is known ---
        # At this point, the missing predicate (is-locked drawer1) should be discovered.
        # You can now proceed with the main plan (e.g., open the drawer if unlocked, etc.)
        # For this exploration, we stop here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()