# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to Discover Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to use available skills to explore the environment and infer what is missing.
        # We will try each skill in a safe, exploratory way and observe the environment for changes.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # Track which predicates or state changes are observed after each skill
        exploration_results = {}

        # Try to move to each location (if possible) and observe
        # Try to pick/place/pull/rotate objects and observe
        # This is a generic exploration loop; adapt as needed for your environment

        print("=== [Exploration Phase] ===")
        for skill in available_skills:
            try:
                print(f"[Exploration] Trying skill: {skill}")
                # For each skill, attempt a generic call with available objects/positions
                # (You may need to adapt argument names/types based on your environment and skill signatures)
                if skill == 'move':
                    # Try moving between two different locations if possible
                    locs = [k for k, v in positions.items() if 'location' in k or 'room' in k or 'pos' in k]
                    if len(locs) >= 2:
                        from_loc = locs[0]
                        to_loc = locs[1]
                        print(f"[Exploration] Moving from {from_loc} to {to_loc}")
                        obs, reward, done = move(env, task, from_loc, to_loc)
                        exploration_results['move'] = (obs, reward, done)
                elif skill == 'pick':
                    # Try picking up an object at a location
                    objs = [k for k in positions.keys() if 'object' in k or 'drawer' in k or 'ball' in k or 'obj' in k]
                    locs = [k for k in positions.keys() if 'location' in k or 'room' in k or 'pos' in k]
                    if objs and locs:
                        obj = objs[0]
                        loc = locs[0]
                        print(f"[Exploration] Picking {obj} at {loc}")
                        obs, reward, done = pick(env, task, obj, loc)
                        exploration_results['pick'] = (obs, reward, done)
                elif skill == 'place':
                    # Try placing an object at a location
                    objs = [k for k in positions.keys() if 'object' in k or 'drawer' in k or 'ball' in k or 'obj' in k]
                    locs = [k for k in positions.keys() if 'location' in k or 'room' in k or 'pos' in k]
                    if objs and locs:
                        obj = objs[0]
                        loc = locs[0]
                        print(f"[Exploration] Placing {obj} at {loc}")
                        obs, reward, done = place(env, task, obj, loc)
                        exploration_results['place'] = (obs, reward, done)
                elif skill == 'rotate':
                    # Try rotating the gripper to a target quaternion (use identity or 90 deg as example)
                    # Use a default gripper and angle if available
                    target_quat = np.array([0, 0, 0, 1])  # Identity quaternion
                    print(f"[Exploration] Rotating gripper to {target_quat}")
                    obs, reward, done = rotate(env, task, target_quat)
                    exploration_results['rotate'] = (obs, reward, done)
                elif skill == 'pull':
                    # Try pulling an object (e.g., a drawer) at a location
                    objs = [k for k in positions.keys() if 'drawer' in k or 'object' in k or 'obj' in k]
                    locs = [k for k in positions.keys() if 'location' in k or 'room' in k or 'pos' in k]
                    if objs and locs:
                        obj = objs[0]
                        loc = locs[0]
                        print(f"[Exploration] Pulling {obj} at {loc}")
                        obs, reward, done = pull(env, task, obj, loc)
                        exploration_results['pull'] = (obs, reward, done)
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed: {e}")

        print("=== [Exploration Results] ===")
        for skill, result in exploration_results.items():
            obs, reward, done = result
            print(f"Skill: {skill}, Reward: {reward}, Done: {done}")
            # Optionally, print or log observation details to help identify missing predicates
            # print(f"Observation after {skill}: {obs}")

        print("=== [End of Exploration Phase] ===")
        print("If the planner still fails, check which predicates are not being updated or observed after these actions.")
        print("You may need to add or correct predicates in the domain/problem PDDL.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here using the available skills.
        # For this code, the focus is on the exploration phase to help identify missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()