# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Error] Failed to get object positions:", e)
            positions = {}

        # === Get object list from positions ===
        object_list = list(positions.keys())
        print("[Info] Objects in environment:", object_list)

        # === Exploration Phase: Identify Missing Predicate ===
        # The exploration domain suggests that some knowledge predicates (e.g., identified, temperature-known, weight-known, durability-known, lock-known) may be missing.
        # We will attempt to explore by moving to each object's location and picking it up, checking for any missing knowledge or errors.

        for obj_name in object_list:
            print(f"\n[Exploration] Attempting to interact with object: {obj_name}")
            obj_pos = positions[obj_name]
            try:
                # 1. Move to the object's position
                print(f"[Exploration] Moving to {obj_name} at position {obj_pos}")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended during move to {obj_name}!")
                    return

                # 2. Attempt to pick the object
                print(f"[Exploration] Attempting to pick {obj_name}")
                try:
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during pick of {obj_name}!")
                        return
                except Exception as pick_exc:
                    print(f"[Exploration] Pick failed for {obj_name}: {pick_exc}")
                    continue

                # 3. Attempt to place the object back (if place skill is available)
                if 'place' in globals():
                    print(f"[Exploration] Attempting to place {obj_name} back at {obj_pos}")
                    try:
                        obs, reward, done = place(
                            env,
                            task,
                            target_pos=np.array(obj_pos),
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Task ended during place of {obj_name}!")
                            return
                    except Exception as place_exc:
                        print(f"[Exploration] Place failed for {obj_name}: {place_exc}")
                        continue

            except Exception as exc:
                print(f"[Exploration] Exception during interaction with {obj_name}: {exc}")
                continue

        # === Exploration for Drawer/Rotational Actions (if present) ===
        # If there are drawers or objects that require rotate/pull, attempt those actions as well.
        for obj_name in object_list:
            if 'drawer' in obj_name.lower():
                print(f"\n[Exploration] Attempting to interact with drawer: {obj_name}")
                obj_pos = positions[obj_name]
                try:
                    # Move to drawer position
                    print(f"[Exploration] Moving to drawer {obj_name} at {obj_pos}")
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during move to {obj_name}!")
                        return

                    # Try rotate (if available)
                    if 'rotate' in globals():
                        print(f"[Exploration] Attempting to rotate {obj_name}")
                        try:
                            # Example: rotate(env, task, gripper_name, from_angle, to_angle)
                            # Here, we use placeholder values; in a real scenario, use actual gripper and angle names.
                            obs, reward, done = rotate(
                                env,
                                task,
                                gripper_name='gripper',
                                from_angle='zero_deg',
                                to_angle='ninety_deg'
                            )
                            if done:
                                print(f"[Exploration] Task ended during rotate of {obj_name}!")
                                return
                        except Exception as rotate_exc:
                            print(f"[Exploration] Rotate failed for {obj_name}: {rotate_exc}")

                    # Try pull (if available)
                    if 'pull' in globals():
                        print(f"[Exploration] Attempting to pull {obj_name}")
                        try:
                            # Example: pull(env, task, gripper_name, drawer_name)
                            obs, reward, done = pull(
                                env,
                                task,
                                gripper_name='gripper',
                                drawer_name=obj_name
                            )
                            if done:
                                print(f"[Exploration] Task ended during pull of {obj_name}!")
                                return
                        except Exception as pull_exc:
                            print(f"[Exploration] Pull failed for {obj_name}: {pull_exc}")

                except Exception as exc:
                    print(f"[Exploration] Exception during drawer interaction with {obj_name}: {exc}")
                    continue

        print("\n[Exploration] Completed exploration of all objects and drawers.")

    finally:
        # Always ensure the environment is properly shutdown
        try:
            shutdown_environment(env)
        except Exception as shutdown_exc:
            print("[Warning] shutdown_environment failed or not defined:", shutdown_exc)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
