# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at anchor-pos-bottom bottom)
        # The missing predicate is likely related to the anchor position of the drawer.
        # We need to explore and check if the anchor position is correctly set in the environment.

        print("[Exploration] Checking for missing predicate: (at anchor-pos-bottom bottom)")
        # Try to access the anchor position of the drawer (if exists)
        anchor_pos_name = 'anchor-pos-bottom'
        bottom_name = 'bottom'
        anchor_pos = positions.get(anchor_pos_name, None)
        bottom_pos = positions.get(bottom_name, None)

        if anchor_pos is None or bottom_pos is None:
            print(f"[Exploration] Could not find positions for '{anchor_pos_name}' or '{bottom_name}'.")
            print("[Exploration] Exploration failed: missing object positions.")
        else:
            # Check if the anchor position is at the bottom (within a small threshold)
            dist = np.linalg.norm(np.array(anchor_pos) - np.array(bottom_pos))
            print(f"[Exploration] Distance between '{anchor_pos_name}' and '{bottom_name}': {dist:.4f}")
            if dist < 0.05:
                print(f"[Exploration] Predicate (at {anchor_pos_name} {bottom_name}) holds (distance < 0.05).")
            else:
                print(f"[Exploration] Predicate (at {anchor_pos_name} {bottom_name}) does NOT hold (distance >= 0.05).")
            # This exploration helps confirm the missing predicate in the environment.

        # === Example Task Plan ===
        # For demonstration, let's try to pick an object at the anchor position if available.
        # Use only predefined skills: pick, place, move, rotate, pull

        # Let's assume we want to pick an object called 'target_object' at 'anchor-pos-bottom'
        target_object_name = 'target_object'
        if anchor_pos is not None and target_object_name in positions:
            target_object_pos = positions[target_object_name]
            print(f"[Task] Attempting to pick '{target_object_name}' at anchor position: {anchor_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=anchor_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Task ended after picking '{target_object_name}'!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")
        else:
            print(f"[Task] Cannot pick: '{target_object_name}' or anchor position not found in positions.")

        # You can add more steps here using other available skills (place, move, rotate, pull) as needed.
        # For example, to place the object at another location:
        # if 'place_location' in positions:
        #     place_pos = positions['place_location']
        #     try:
        #         obs, reward, done = place(
        #             env,
        #             task,
        #             target_pos=place_pos,
        #             approach_distance=0.15,
        #             max_steps=100,
        #             threshold=0.01,
        #             approach_axis='z',
        #             timeout=10.0
        #         )
        #         if done:
        #             print(f"[Task] Task ended after placing at 'place_location'!")
        #             return
        #     except Exception as e:
        #         print(f"[Task] Exception during place: {e}")

        # TODO: Add further plan steps as required by your specific oracle plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
