# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer1': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (is-open drawer1) ===
        # Feedback indicates we need to check if drawer1 is open.
        # We'll use available skills to explore the state of drawer1.

        # Assumptions:
        # - 'drawer1' is a key in positions, representing the drawer's handle or front.
        # - The gripper must move to the drawer, rotate if needed, and attempt to pull.
        # - If the drawer is already open, the pull action may have no effect or may fail gracefully.

        # Step 1: Move gripper to the front of drawer1
        try:
            drawer1_pos = positions['drawer1']
        except KeyError:
            print("[Error] 'drawer1' position not found in object_positions.")
            return

        print("[Exploration] Moving to drawer1 position:", drawer1_pos)
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(drawer1_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to drawer1!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            return

        # Step 2: Rotate gripper if required (assume 'ninety_deg' is needed for side approach)
        # We assume the rotate skill takes (env, task, from_angle, to_angle)
        # For demonstration, we use string names as angles, e.g., 'zero_deg', 'ninety_deg'
        # These should match the PDDL and environment conventions.
        try:
            print("[Exploration] Rotating gripper to ninety_deg.")
            obs, reward, done = rotate(
                env,
                task,
                from_angle='zero_deg',
                to_angle='ninety_deg'
            )
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")
            # If rotate is not needed or fails, continue

        # Step 3: Attempt to pull the drawer open
        # The pull skill may require the gripper to be holding the drawer handle.
        # If not, we may need to pick or grasp the handle first.
        # For this exploration, we attempt pull directly and handle exceptions.
        try:
            print("[Exploration] Attempting to pull drawer1.")
            obs, reward, done = pull(
                env,
                task,
                drawer_name='drawer1'
            )
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            # If pull fails, the drawer may already be open or not grasped; exploration ends here.

        # Step 4: Check if the drawer is open (missing predicate)
        # In a real system, we would check the environment state or observation for 'is-open drawer1'
        # For this skeleton, we print a message indicating the exploration is complete.
        print("[Exploration] Checked (is-open drawer1) via pull action.")

        # === End of Exploration Phase ===

        # (Optional) Continue with the main task plan if needed
        # For this task, the focus is on exploration and predicate discovery.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()