# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to explore the environment to determine which predicate is missing.
        # For example, we may need to check if the robot can identify an object's temperature, weight, or durability.
        # We'll attempt to use the available skills to interact with objects and observe the results.

        # For demonstration, let's iterate over all objects and try to pick and place them,
        # observing if any information (e.g., weight-known, durability-known) is missing or not updated.

        # List of objects to explore (replace with actual keys from positions)
        object_keys = list(positions.keys())
        print(f"[Exploration] Objects found in environment: {object_keys}")

        # For each object, try to pick and place it, and log any issues or missing predicates
        for obj_name in object_keys:
            try:
                obj_pos = positions[obj_name]
                print(f"[Exploration] Attempting to pick object: {obj_name} at position {obj_pos}")

                # Move to the object's position
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly while moving to {obj_name}.")
                    return

                # Attempt to pick the object
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly while picking {obj_name}.")
                    return

                # Attempt to place the object back at its original position (or a default place position)
                place_pos = obj_pos  # For simplicity, place at the same position
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(place_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly while placing {obj_name}.")
                    return

                print(f"[Exploration] Successfully picked and placed {obj_name}.")

            except Exception as e:
                print(f"[Exploration] Exception occurred while handling {obj_name}: {e}")

        # === Exploration for Drawer/Rotational Actions (if present) ===
        # If there are drawers or objects requiring rotate/pull, attempt those actions as well.
        # For demonstration, check for known drawer or handle keys in positions.
        drawer_keys = [k for k in positions if 'drawer' in k or 'handle' in k]
        for drawer_name in drawer_keys:
            try:
                drawer_pos = positions[drawer_name]
                print(f"[Exploration] Attempting to rotate and pull drawer: {drawer_name} at position {drawer_pos}")

                # Move to the drawer's position
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(drawer_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly while moving to {drawer_name}.")
                    return

                # Attempt to rotate (if applicable)
                try:
                    obs, reward, done = rotate(
                        env,
                        task,
                        target_pos=np.array(drawer_pos),
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended unexpectedly during rotate on {drawer_name}.")
                        return
                    print(f"[Exploration] Rotated {drawer_name}.")
                except Exception as e:
                    print(f"[Exploration] Rotate not applicable or failed for {drawer_name}: {e}")

                # Attempt to pull (if applicable)
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=np.array(drawer_pos),
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended unexpectedly during pull on {drawer_name}.")
                        return
                    print(f"[Exploration] Pulled {drawer_name}.")
                except Exception as e:
                    print(f"[Exploration] Pull not applicable or failed for {drawer_name}: {e}")

            except Exception as e:
                print(f"[Exploration] Exception occurred while handling {drawer_name}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()