# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback and exploration knowledge indicate we must check for object type (e.g., tomato)
        # and ensure safety/force calibration before picking, and check placement after placing.
        # We also need to ensure at least two tomatoes are available.

        # 1. Find all tomatoes in the environment
        tomato_names = [name for name in positions if 'tomato' in name]
        if len(tomato_names) < 2:
            print("[Error] Less than two tomatoes found in the environment. Aborting task.")
            return

        # 2. Check that the objects are actually tomatoes (simulate identification)
        identified_tomatoes = []
        for tname in tomato_names:
            # Simulate identification (in real system, would use vision or predicate)
            if 'tomato' in tname:
                identified_tomatoes.append(tname)
        if len(identified_tomatoes) < 2:
            print("[Error] Less than two identified tomatoes. Aborting task.")
            return

        # 3. Safety/Force Calibration (simulate by checking for 'weight-known' or similar)
        # In this code, we simulate this by attempting a pick and catching exceptions

        # 4. Pick up the first tomato
        tomato1 = identified_tomatoes[0]
        tomato1_pos = positions[tomato1]
        print(f"[Task] Attempting to pick up {tomato1} at position {tomato1_pos}")

        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to pick up {tomato1}: {e}")
            return

        if done:
            print("[Task] Task ended after picking up first tomato!")
            return

        # 5. Place the first tomato on the plate (assume plate is present in positions)
        plate_name = None
        for name in positions:
            if 'plate' in name:
                plate_name = name
                break
        if plate_name is None:
            print("[Error] Plate not found in environment. Aborting task.")
            return
        plate_pos = positions[plate_name]
        print(f"[Task] Placing {tomato1} on plate at position {plate_pos}")

        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to place {tomato1} on plate: {e}")
            return

        if done:
            print("[Task] Task ended after placing first tomato!")
            return

        # 6. Check if the tomato is on the plate (simulate by checking position)
        obs = task.get_observation()
        # In a real system, would check with sensors or vision; here, we simulate
        tomato1_new_pos = positions.get(tomato1, None)
        if tomato1_new_pos is None:
            print(f"[Warning] Could not retrieve new position for {tomato1}.")
        else:
            dist_to_plate = np.linalg.norm(np.array(tomato1_new_pos) - np.array(plate_pos))
            if dist_to_plate > 0.10:
                print(f"[Warning] {tomato1} not placed close enough to plate (distance: {dist_to_plate:.3f}).")
            else:
                print(f"[Check] {tomato1} successfully placed on plate.")

        # 7. Repeat for the second tomato
        tomato2 = identified_tomatoes[1]
        tomato2_pos = positions[tomato2]
        print(f"[Task] Attempting to pick up {tomato2} at position {tomato2_pos}")

        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to pick up {tomato2}: {e}")
            return

        if done:
            print("[Task] Task ended after picking up second tomato!")
            return

        print(f"[Task] Placing {tomato2} on plate at position {plate_pos}")

        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to place {tomato2} on plate: {e}")
            return

        if done:
            print("[Task] Task ended after placing second tomato!")
            return

        # 8. Check if the second tomato is on the plate
        obs = task.get_observation()
        tomato2_new_pos = positions.get(tomato2, None)
        if tomato2_new_pos is None:
            print(f"[Warning] Could not retrieve new position for {tomato2}.")
        else:
            dist_to_plate = np.linalg.norm(np.array(tomato2_new_pos) - np.array(plate_pos))
            if dist_to_plate > 0.10:
                print(f"[Warning] {tomato2} not placed close enough to plate (distance: {dist_to_plate:.3f}).")
            else:
                print(f"[Check] {tomato2} successfully placed on plate.")

        print("[Task] Successfully picked and placed two tomatoes on the plate.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
