# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        object_list = list(positions.keys())

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may not know some property (e.g., lock-known, identified, etc.)
        # We'll attempt to "move" to each location and "pick" each object, checking for errors or missing predicates.

        # For demonstration, let's try to pick each object at its location, and handle errors if the object is not present.
        for obj_name in object_list:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to pick object '{obj_name}' at position {obj_pos}.")

            # Error Handling: Check if object is in the actual object list
            if obj_name not in object_list:
                print(f"Error: Object '{obj_name}' not found in the actual object list.")
                continue

            # Safety Feature: Check if the gripper's pose is safe before moving
            obs = task.get_observation()
            gripper_pose = getattr(obs, 'gripper_pose', [0,0,0,0,0,0,0])
            if gripper_pose[0] > 0.5 or gripper_pose[0] < -0.5:
                print("Error: Gripper pose is not safe. Aborting pick for this object.")
                continue

            # Try to pick the object using the predefined skill
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked object '{obj_name}'.")
                if done:
                    print("[Exploration] Task ended after picking object!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pick of '{obj_name}': {e}")
                continue

            # Force Calibration: (if available in obs)
            force_calibrated = False
            if hasattr(obs, 'gripper_force_sensor'):
                while not force_calibrated:
                    if obs.gripper_force_sensor > 0.5:
                        force_calibrated = True
                        print("[Exploration] Gripper force sensor calibrated.")
                    else:
                        # Try to recalibrate by repeating the action
                        obs, reward, done = pick(
                            env,
                            task,
                            target_pos=obj_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print("[Exploration] Task ended during force calibration!")
                            return

            # Logic Refinement: Check if the pick was successful
            if hasattr(obs, 'holding') and not obs.holding:
                print(f"Error: Pick failed for '{obj_name}'. Aborting.")
                continue

            # Place the object back (if possible)
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Placed object '{obj_name}' back.")
                if done:
                    print("[Exploration] Task ended after placing object!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during place of '{obj_name}': {e}")
                continue

        # === Exploration for Drawer/Rotation/Other Skills ===
        # If there are drawers or objects that require rotation or pulling, attempt those as well.
        # For demonstration, try to rotate each object if the skill is available.
        for obj_name in object_list:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to rotate object '{obj_name}'.")

            # Example: Try to rotate to a target quaternion (identity quaternion as placeholder)
            target_quat = np.array([0, 0, 0, 1])
            try:
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=50,
                    threshold=0.05,
                    timeout=5.0
                )
                print(f"[Exploration] Rotated object '{obj_name}'.")
                if done:
                    print("[Exploration] Task ended after rotation!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during rotation of '{obj_name}': {e}")
                continue

        # === Exploration for Pull Skill ===
        # If the object is a drawer or can be pulled, try the pull skill
        for obj_name in object_list:
            print(f"[Exploration] Attempting to pull object '{obj_name}'.")
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    obj_name
                )
                print(f"[Exploration] Pulled object '{obj_name}'.")
                if done:
                    print("[Exploration] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pull of '{obj_name}': {e}")
                continue

        print("[Exploration] Completed exploration phase. If any missing predicates were encountered, check logs above.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()