# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding gripper1) -- This is not a valid predicate in the domain.
        # The correct predicate should be (holding ?obj), where ?obj is an object, not a gripper.
        # Therefore, we need to explore and confirm the correct predicate by attempting to pick an object.

        # Find an object to pick (excluding grippers, drawers, etc.)
        # We'll assume the positions dict contains at least one pickable object.
        # For demonstration, pick the first object that is not a gripper or drawer.
        pickable_obj_name = None
        pickable_obj_pos = None
        for obj_name, obj_pos in positions.items():
            if "gripper" not in obj_name and "drawer" not in obj_name:
                pickable_obj_name = obj_name
                pickable_obj_pos = obj_pos
                break

        if pickable_obj_name is None:
            print("[Exploration] No pickable object found in positions. Exploration cannot proceed.")
        else:
            print(f"[Exploration] Attempting to pick object: {pickable_obj_name} at {pickable_obj_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(pickable_obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                # After pick, check if the correct predicate is achieved
                # The correct predicate should be (holding {pickable_obj_name})
                # If the feedback was (holding gripper1), that is incorrect.
                print(f"[Exploration] Picked {pickable_obj_name}. Now checking predicates/state...")
                # (In a real system, you would check the internal state or observation for the correct predicate.)
            except Exception as e:
                print(f"[Exploration] Exception during pick: {e}")

        # === Main Task Plan Execution ===
        # Here, you would continue with the oracle plan using only the available skills:
        # ['pick', 'place', 'move', 'rotate', 'pull']
        # For demonstration, let's try to place the object if it was picked.

        if pickable_obj_name is not None and pickable_obj_pos is not None:
            # Find a place location (for demonstration, offset the original position)
            place_offset = np.array([0.2, 0, 0])  # Move 20cm in x
            place_pos = np.array(pickable_obj_pos) + place_offset
            print(f"[Task] Attempting to place {pickable_obj_name} at {place_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Task] Placed {pickable_obj_name} at {place_pos}")
                if done:
                    print("[Task] Task ended after placing object!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")

        # === Additional Plan Steps (if any) ===
        # You can add more steps here using the available skills as needed.
        # For example, rotate, pull, or move actions if required by the oracle plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()