# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at gripper nowhere-pos)
        # The missing predicate is likely related to the gripper's position or state.
        # We will perform an exploration step to check the gripper's position and log it.

        print("[Exploration] Checking gripper position for missing predicate...")

        try:
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            print(f"[Exploration] Gripper position: {gripper_pos}")
        except Exception as e:
            print(f"[Exploration] Could not retrieve gripper position: {e}")

        # If the feedback is about (at gripper nowhere-pos), we may want to check if the gripper is at a known position.
        # This can be used to inform further planning or debugging.

        # === Example Task Plan ===
        # For demonstration, we will attempt to pick and place an object if available.
        # Use only the provided skills: pick, place, move, rotate, pull

        # Find an object to pick (if any)
        object_to_pick = None
        target_pos = None
        for obj_name, pos in positions.items():
            if "gripper" not in obj_name and "drawer" not in obj_name:
                object_to_pick = obj_name
                target_pos = pos
                break

        if object_to_pick is not None and target_pos is not None:
            print(f"[Task] Attempting to pick object: {object_to_pick} at position {target_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=target_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")

            # Attempt to place the object at a new location (if available)
            # For demonstration, try to place at a position offset from the pick position
            place_offset = np.array([0.2, 0, 0])
            place_pos = np.array(target_pos) + place_offset
            print(f"[Task] Attempting to place object: {object_to_pick} at position {place_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
        else:
            print("[Task] No suitable object found to pick and place.")

        # === Additional Exploration for Drawer/Gripper ===
        # If there is a drawer and gripper, try to rotate or pull as an exploration
        drawer_name = None
        gripper_name = None
        for obj_name in positions.keys():
            if "drawer" in obj_name:
                drawer_name = obj_name
            if "gripper" in obj_name:
                gripper_name = obj_name

        if drawer_name and gripper_name:
            print(f"[Exploration] Found drawer: {drawer_name} and gripper: {gripper_name}")
            # Try to rotate the gripper if possible
            try:
                # For demonstration, rotate from 'zero_deg' to 'ninety_deg' if those angles exist
                from_angle = 'zero_deg'
                to_angle = 'ninety_deg'
                print(f"[Exploration] Attempting to rotate gripper {gripper_name} from {from_angle} to {to_angle}")
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name,
                    from_angle,
                    to_angle
                )
                if done:
                    print("[Exploration] Task ended after rotate!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during rotate: {e}")

            # Try to pull the drawer if possible
            try:
                print(f"[Exploration] Attempting to pull drawer {drawer_name} with gripper {gripper_name}")
                obs, reward, done = pull(
                    env,
                    task,
                    gripper_name,
                    drawer_name
                )
                if done:
                    print("[Exploration] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pull: {e}")

        print("[Task] Exploration and main task steps completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()