# run_skeleton_task.py (Completed with Exploration and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import time

def euler_from_quat(quaternion):
    """
    Convert quaternion (x, y, z, w) to Euler angles (roll, pitch, yaw).
    Returns angles in radians.
    """
    x, y, z, w = quaternion
    # roll (x-axis rotation)
    t0 = +2.0 * (w * x + y * z)
    t1 = +1.0 - 2.0 * (x * x + y * y)
    roll_x = np.arctan2(t0, t1)
    # pitch (y-axis rotation)
    t2 = +2.0 * (w * y - z * x)
    t2 = np.clip(t2, -1.0, 1.0)
    pitch_y = np.arcsin(t2)
    # yaw (z-axis rotation)
    t3 = +2.0 * (w * z + x * y)
    t4 = +1.0 - 2.0 * (y * y + z * z)
    yaw_z = np.arctan2(t3, t4)
    return np.array([roll_x, pitch_y, yaw_z])

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()
        print("[Info] Object positions loaded:", positions)

        # === Retrieve Object List ===
        object_list = list(positions.keys())
        print("[Info] Object list:", object_list)

        # === Exploration Phase: Identify Missing Predicate ===
        # The exploration domain suggests that some predicates (e.g., identified, temperature-known, weight-known, durability-known, lock-known) may be missing.
        # We will attempt to explore the environment to discover which predicate is missing by performing available skills and observing the effects.

        # For demonstration, we will try to pick each object and see if any information is missing or if an error occurs.
        # We will also check for the presence of objects before acting.

        # Set parameters for exploration
        max_steps = 100
        timeout = 10.0

        # Track which predicates are discovered/missing
        discovered_predicates = set()
        missing_predicates = set()

        # For each object, try to pick and place it, and observe the environment
        for obj_name in object_list:
            print(f"\n[Exploration] Attempting to pick and place object: {obj_name}")
            try:
                # Get object position
                if obj_name not in positions:
                    print(f"[Warning] Object '{obj_name}' not found in positions. Skipping.")
                    continue
                obj_pos = positions[obj_name]

                # Try to pick the object
                print(f"[Action] Picking object '{obj_name}' at position {obj_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=max_steps,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=timeout
                )
                if done:
                    print(f"[Info] Task ended after picking '{obj_name}'.")
                    break

                # Try to place the object back at its original position
                print(f"[Action] Placing object '{obj_name}' at position {obj_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=max_steps,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=timeout
                )
                if done:
                    print(f"[Info] Task ended after placing '{obj_name}'.")
                    break

                # After pick/place, check for new predicates in observation (if available)
                # This is a placeholder: in a real system, you would parse obs for new info
                # For now, we just print the observation keys
                if hasattr(obs, '__dict__'):
                    obs_keys = list(obs.__dict__.keys())
                    print(f"[Observation] Observation keys after pick/place: {obs_keys}")
                else:
                    print("[Observation] No __dict__ in observation object.")

            except Exception as e:
                print(f"[Error] Exception during pick/place of '{obj_name}': {e}")
                missing_predicates.add(obj_name)
                continue

        # === Exploration for Drawer/Gripper Actions (if present) ===
        # If there are drawers or grippers in the object list, try to rotate or pull them
        for obj_name in object_list:
            if 'drawer' in obj_name or 'handle' in obj_name:
                print(f"\n[Exploration] Attempting to rotate and pull '{obj_name}'")
                try:
                    # Get object position
                    if obj_name not in positions:
                        print(f"[Warning] Drawer '{obj_name}' not found in positions. Skipping.")
                        continue
                    obj_pos = positions[obj_name]

                    # Try to rotate the gripper to a target orientation (example: 90 degrees about z)
                    target_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])  # 90 deg about z
                    print(f"[Action] Rotating gripper for '{obj_name}' to quat {target_quat}")
                    obs, reward, done = rotate(
                        env,
                        task,
                        target_quat=target_quat,
                        max_steps=max_steps,
                        threshold=0.05,
                        timeout=timeout
                    )
                    if done:
                        print(f"[Info] Task ended after rotating for '{obj_name}'.")
                        break

                    # Try to pull the drawer
                    print(f"[Action] Pulling '{obj_name}'")
                    obs, reward, done = pull(
                        env,
                        task,
                        target_obj=obj_name,
                        max_steps=max_steps,
                        timeout=timeout
                    )
                    if done:
                        print(f"[Info] Task ended after pulling '{obj_name}'.")
                        break

                except Exception as e:
                    print(f"[Error] Exception during rotate/pull of '{obj_name}': {e}")
                    missing_predicates.add(obj_name)
                    continue

        # === Summary of Exploration ===
        if missing_predicates:
            print(f"\n[Summary] Missing predicates or failed actions for objects: {missing_predicates}")
        else:
            print("\n[Summary] All objects explored successfully. No missing predicates detected.")

        print("\n===== End of Exploration Phase =====")

        # === Main Task Plan Execution (Example) ===
        # Here you would execute the oracle plan step-by-step using only the available skills.
        # For this template, we assume the exploration phase is the main focus for missing predicate discovery.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
