# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration knowledge suggest that the robot may be missing knowledge about an object's property (e.g., lock-known, weight-known, durability-known, etc.)
        # We will perform an exploration phase to try to discover which predicate is missing by using available skills and observing the environment.

        # For demonstration, let's assume we have a drawer and an object to manipulate.
        # We'll try to interact with them and see if any knowledge is missing (e.g., is the drawer locked? is the object too heavy? etc.)

        # Replace these with actual object/location names from your environment
        drawer_name = None
        object_name = None
        drawer_handle_pos = None
        object_pos = None

        # Try to find a drawer and an object from positions
        for name in positions:
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_handle_pos = positions[name]
            elif 'object' in name or 'ball' in name or 'item' in name:
                object_name = name
                object_pos = positions[name]

        # Fallback: just pick the first two if not found
        if drawer_name is None and len(positions) > 0:
            drawer_name = list(positions.keys())[0]
            drawer_handle_pos = positions[drawer_name]
        if object_name is None and len(positions) > 1:
            object_name = list(positions.keys())[1]
            object_pos = positions[object_name]

        # === Exploration: Try to pick the object and see if any predicate is missing ===
        if object_name and object_pos is not None:
            print(f"[Exploration] Attempting to pick object: {object_name} at {object_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=object_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after pick attempt!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pick: {e}")

        # === Exploration: Try to rotate the gripper at the drawer handle position ===
        if drawer_handle_pos is not None:
            print(f"[Exploration] Attempting to rotate at drawer handle: {drawer_name} at {drawer_handle_pos}")
            try:
                # For demonstration, rotate to a fixed quaternion (e.g., 90 degrees about z)
                target_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])  # 90 deg about z
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=100,
                    threshold=0.05,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after rotate attempt!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during rotate: {e}")

        # === Exploration: Try to pull the drawer (if possible) ===
        if drawer_handle_pos is not None:
            print(f"[Exploration] Attempting to pull drawer: {drawer_name}")
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    target_pos=drawer_handle_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='y',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after pull attempt!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pull: {e}")

        # === Exploration: Try to place the object (if holding) ===
        if object_name and object_pos is not None:
            print(f"[Exploration] Attempting to place object: {object_name} at {object_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=object_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after place attempt!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during place: {e}")

        # === Exploration Summary ===
        print("[Exploration] Exploration phase complete. If any action failed, check logs for missing predicates or knowledge.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
